import cloneDeep from "lodash/cloneDeep";
import moment from 'moment';
export default class ScheduleReservationClass {
    constructor(objReservation, objServiceSettings) {
        this.reservation = objReservation;
        this.service_settings = objServiceSettings;
    }

    static setCustomerReservation(customerData) {
        let bookingObj = cloneDeep(default_booking);
        bookingObj.customer = {
            id: customerData.id,
            first_name: customerData.first_name,
            last_name: customerData.last_name,
            mobile_number: customerData.mobile_number,
            phone_number: customerData.phone_number,
            email: customerData.email,
            cc: {
                type: customerData.card_type,
                number: customerData.cc_number,
                expiry: customerData.cc_expiry,
                status: customerData.cc_status,
            },
        };
        bookingObj.reservation.pickup_location = customerData.address
            ? customerData.address
            : "";
        bookingObj.payment_info.payment_type = customerData.pay_type_id
            ? customerData.pay_type_id
            : "";
        return bookingObj;
    }
    static setReservationFromRaw(rawData) {
        let bookingObj = cloneDeep(default_booking);
        bookingObj.service_settings.reservation_id = rawData.id;
        bookingObj.service_settings.service_type_id = rawData.service_type_id;
        bookingObj.service_settings.is_edit = true;
        //bookingObj.service_settings.is_round_trip = rawData.round_trip;
        bookingObj.service_settings.customer_notes = rawData.customer_notes;
        bookingObj.customer = {
            id: rawData.customer.id,
            first_name: rawData.customer.first_name,
            last_name: rawData.customer.last_name,
            mobile_number: rawData.customer.mobile_number,
            phone_number: rawData.customer.phone_number,
            email: rawData.customer.email,
            cc: {
                type: rawData.customer.card_type,
                number: rawData.customer.cc_number,
                expiry: rawData.customer.cc_expiry,
                status: rawData.customer.cc_status,
            },
        };
        bookingObj.payment_info.payment_type = rawData.customer.pay_type_id
            ? rawData.customer.pay_type_id
            : "";

        bookingObj.reservation.id = rawData.id;
        bookingObj.reservation.pickup_date = rawData.pick_up_date;
        bookingObj.reservation.pickup_time = rawData.pick_up_time;
        bookingObj.reservation.pickup_location = rawData.pick_up_location;
        bookingObj.reservation.dropoff_location = rawData.drop_off_location;

        bookingObj.reservation.time_duration_text = rawData.time_duration_text;
        bookingObj.reservation.distance = rawData.distance;
        bookingObj.reservation.time_duration_hours =
            rawData.time_duration_hours;
        bookingObj.reservation.ride_type = rawData.ride_type;
        bookingObj.reservation.additional_stops = rawData.additional_stops
            ? rawData.additional_stops
            : [];
        bookingObj.reservation.travellers = rawData.travellers_types;
        bookingObj.reservation.luggage = rawData.luggage;
        bookingObj.reservation.drop_off_hours = rawData.drop_off_hours;
        bookingObj.reservation.is_on_call = rawData.is_on_call
            ? rawData.is_on_call
            : false;

        bookingObj.reservation.admin_discount = rawData.admin_discount;
        bookingObj.reservation.driver_gratuity = rawData.driver_gratuity;
        //bookingObj.reservation.reservationCity = rawData.city_id;

        //bookingObj.reservation.selected_vehicle = rawData.selected_vehicle;
        bookingObj.reservation.selected_vehicle = {
            id: rawData.vehicle_type_id,
            ride_type: rawData.ride_type,
            passengers_capacity: rawData.selected_vehicle?.passengers_capacity,
            schedule_fare_detail: rawData.schedule_fare_detail,
            selectedServiceId:rawData.selected_vehicle.selectedServiceId,
        };
        bookingObj.reservation.timeLimitError = true;
        bookingObj.reservation.reservationCity = {
            id: rawData.city_id,
        };

        return bookingObj;
    }
    getReservationFromRaw(rawData) {
        let bookingObj = cloneDeep(default_booking);
        bookingObj.service_settings.service_type_id = rawData.service_type_id;
        //bookingObj.service_settings.is_round_trip = rawData.round_trip;
        bookingObj.service_settings.customer_notes = rawData.customer_notes;
        bookingObj.customer = rawData.customer;
        bookingObj.reservation.pickup_date = rawData.pick_up_date;
        bookingObj.reservation.pickup_time = rawData.pick_up_time;
        bookingObj.reservation.pickup_location = rawData.pick_up_location;
        bookingObj.reservation.dropoff_location = rawData.drop_off_location;

        bookingObj.reservation.time_duration_text = rawData.time_duration_text;
        bookingObj.reservation.distance = rawData.distance;
        bookingObj.reservation.time_duration_hours =
            rawData.time_duration_hours;
        bookingObj.reservation.ride_type = rawData.ride_type;
        bookingObj.reservation.additional_stops = rawData.additional_stops
            ? rawData.additional_stops
            : [];
        bookingObj.reservation.travellers = rawData.travellers_types;
        bookingObj.reservation.luggage = rawData.luggage;
        bookingObj.reservation.drop_off_hours = rawData.drop_off_hours;
        bookingObj.reservation.is_on_call = rawData.is_on_call
            ? rawData.is_on_call
            : false;

        bookingObj.reservation.admin_discount = rawData.admin_discount;
        bookingObj.reservation.driver_gratuity = rawData.driver_gratuity;
        bookingObj.reservation.reservationCity = rawData.city_id;
        bookingObj.reservation.total_fare = rawData.total_fare;

        //bookingObj.reservation.selected_vehicle = rawData.selected_vehicle;
        bookingObj.reservation.selected_vehicle = {
            id: rawData.vehicle_type_id,
            ride_type: rawData.ride_type,
            passengers_capacity: rawData.selected_vehicle?.passengers_capacity,
            schedule_fare_detail: rawData.schedule_fare_detail,
        };

        return new ScheduleReservationClass(
            bookingObj.reservation,
            bookingObj.service_settings
        );
    }
    getDepartureTimeLimit(objVehicle) {
        let objVehicleFareDetail = {};
        if (objVehicle.ride_type === "hourly") {
            objVehicleFareDetail = objVehicle.hourly_fare_detail;
        } else if (objVehicle.ride_type === "zip_code") {
            objVehicleFareDetail = objVehicle.milage_fare_detail;
        } else {
            objVehicleFareDetail = objVehicle.milage_fare_detail;
        }
        return {
            hours: objVehicleFareDetail.departure_time_limit.pickup_hours,
            minutes: objVehicleFareDetail.departure_time_limit.pickup_minutes,
            total_minutes:
                Number(objVehicleFareDetail.departure_time_limit.pickup_hours) *
                    60 +
                Number(
                    objVehicleFareDetail.departure_time_limit.pickup_minutes
                ),
        };
    }
    total_fare(objVehicle) {
        return (this.reservation.total_fare =
            this.sub_total_fare(objVehicle) -
            (Number(this.reservation.coupon.coupon_price)
                ? Number(this.reservation.coupon.coupon_price)
                : 0));
    }

    sub_total_fare(objVehicle) {
        let objVehicleFareDetail = objVehicle.schedule_fare_detail,
            baseFare = 0;

        const additionalCharges = this.sumAdditionalCharges(
            objVehicleFareDetail.additional_charges
        );
        const timeBaseCharges = objVehicleFareDetail.timely_charges.reduce(
            (acc, charge) => acc + charge.amount,
            0
        );
        const discounts = objVehicleFareDetail.discounts.reduce(
            (acc, charge) => acc + charge.amount,
            0
        );
        return (
            objVehicleFareDetail.base_fare +
            Number(this.reservation.driver_gratuity) +
            (additionalCharges + timeBaseCharges) -
            this.reservation.admin_discount -
            discounts
        );
    }
    getAdditionalCharges(charges) {
        //alert(this.service_settings.service_type_id);
        let filtered_charges = charges
            .filter(
                (charge) =>
                    !(charge.id === 10 && this.reservation.is_on_call === false)
            )
            .filter(
                (charge) =>
                    !(
                        charge.id === 4 &&
                        this.service_settings.service_type_id !== 1
                    )
            )
            .filter(
                (charge) =>
                    !(
                        charge.id === 4 &&
                        this.reservation.flight_info.flight_type !== "arrival"
                    )
            );

        return filtered_charges;
    }
    sumAdditionalCharges(charges) {
        let sum = 0;
        this.getAdditionalCharges(charges).forEach((charge) => {
            sum += charge.amount;
        });
        return sum;
    }

    totalTravellers() {
        return this.reservation.travellers.reduce(
            (total, obj) => total + obj.count,
            0
        );
    }
    totalAdditionStops() {
        return this.reservation.additional_stops.length;
    }
    totalVehicles(vehicle) {
        return Math.ceil(
            this.totalTravellers() / parseInt(vehicle.passengers_capacity)
        );
    }

    totalVehiclesForLuggage(vehicle) {
        if (!this.reservation.luggage || this.reservation.luggage <= 0) {
            return 1; // At least 1 vehicle is needed
        }
        
        const luggageCount = parseInt(this.reservation.luggage);
        const vehicleLuggageCapacity = parseInt(vehicle.luggage_capacity || vehicle.passengers_capacity); // Fallback to passengers_capacity if luggage_capacity not available
        
        // Calculate vehicles needed based on luggage capacity
        return Math.ceil(luggageCount / vehicleLuggageCapacity);
    }

    totalVehiclesRequired(vehicle) {
        const travellersVehicles = this.totalVehicles(vehicle);
        const luggageVehicles = this.totalVehiclesForLuggage(vehicle);
        
        // Return the maximum of both requirements
        return Math.max(travellersVehicles, luggageVehicles);
    }
    getPickupLocation() {
        return this.reservation.pickup_location.address;
    }
    getPickupDateTime() {
        return (
            this.reservation.pickup_time + " " + this.reservation.pickup_date
        );
    }
    getPickupDateTimeMoment() {
        return moment(
            this.reservation.pickup_date + " " + this.reservation.pickup_time,
            "YYYY-MM-DD HH:mm:ss"
        );
    }
    getDropOffLocation() {
        return this.reservation.dropoff_location.address;
    }
    getDistance() {
        return this.reservation.distance;
    }
    getDDropOfHours() {
        return this.reservation.drop_off_hours;
    }
}

export const default_booking = {
    service_settings: {
        reservation_id: 0,
        service_type_id: 4,
        is_round_trip: false,
        customer_notes: "",
        is_admin: true,
        isContinueClicked:false,
        isRoundContinueClicked:false,
        is_edit: false,
    },
    customer: {
        id: 0,
        first_name: "",
        last_name: "",
        mobile_number: "",
        phone_number: "",
        email: "",
    },
    reservation: {
        id: 0,
        customer_picked_date_time: false,
        pickup_date: "",
        pickup_time: "",
        pickup_location: {
            address: "",
        },
        dropoff_location: {
            address: "",
        },
        filterData:{
            pickupAddress: '',
            dropOffAddress: '',
            reservation_date: moment().add(1, 'days').format('YYYY-MM-DD'),
            reservation_day: moment().add(1, 'days').format('ddd'),
            travellers: '',
            searchPickUpQuery: '',
            searchDropOffQuery: '',
        },
        time_duration_text: "",
        distance: 0,
        time_duration_hours: "0",
        ride_type: "schedule",
        additional_stops: [],
        travellers: [
            {
                id: 1,
                name: "Adult",
                count: 1,
            },
        ],
        luggage: 0,
        check_in_bags: 0,
        drop_off_hours: 1,
        scheduled: {
            ride_type: "",
        },
        is_on_call: false,
        selected_vehicle: {},
        admin_discount: 0,
        driver_gratuity: 0,
        total_fare: 0,
        timeLimitError: "",
        reservationCity: "",
        coupon: {
            id: "",
            coupon_code: "",
            coupon_price: "",
            coupon_type: "",
            expiry_date: "",
        },
        "price_per_person":""
    },
    reservation_2: {
        id: 0,
        customer_picked_date_time: false,
        pickup_date: "",
        pickup_time: "",
        pickup_location: {
            address: "",
        },
        dropoff_location: {
            address: "",
        },
        filterData:{
            pickupAddress: '',
            dropOffAddress: '',
            reservation_date: moment().add(1, 'days').format('YYYY-MM-DD'),
            reservation_day: moment().add(1, 'days').format('ddd'),
            travellers: '',
            searchPickUpQuery: '',
            searchDropOffQuery: '',
        },
        time_duration_text: "",
        distance: 0,
        time_duration_hours: "0",
        ride_type: "schedule",
        additional_stops: [],
        travellers: [
            {
                id: 1,
                name: "Adult",
                count: 1,
            },
        ],
        luggage: 0,
        check_in_bags: 0,
        drop_off_hours: 1,
        scheduled: {
            ride_type: "",
        },
        is_on_call: false,
        selected_vehicle: {},
        admin_discount: 0,
        driver_gratuity: 0,
        total_fare: 0,
        timeLimitError: "",
        reservationCity: "",
        coupon: {
            id: "",
            coupon_code: "",
            coupon_price: "",
            coupon_type: "",
            expiry_date: "",
        },
        "price_per_person":""
    },
    payment_info: {
        payment_type: 1,
        use_existing_cc: false,
        cc_info: {
            card_type: "",
            cc_number: "",
            cc_expiry_month: "",
            cc_expiry_year: "",
            csv_code: "",
            cc_status: "",
            billing_zip_code: "",
            card_holder_name: "",
            billing_address: "",
            billing_phone_number: "",
            is_address_same: "",
        },
        direct_account: {
            account_id: 1,
            account_name: "",
            account_number: "",
            account_verified: false,
        },
    },
    coupon: {
        id: "",
        coupon_code: "",
        coupon_price: "",
        coupon_type: "",
        expiry_date: "",
    },
    customer_notes: "",
    completed: false,
};

