import cloneDeep from "lodash/cloneDeep";
import moment from "moment";

export default class ReservationClass {
    constructor(objReservation, objServiceSettings) {
        this.reservation = objReservation;
        this.service_settings = objServiceSettings;
    }

    static setCustomerReservation(customerData) {
        let bookingObj = cloneDeep(default_booking);
        bookingObj.customer = {
            id: customerData.id,
            first_name: customerData.first_name,
            last_name: customerData.last_name,
            mobile_number: customerData.mobile_number,
            phone_number: customerData.phone_number,
            email: customerData.email,
            cc: {
                type: customerData.card_type,
                number: customerData.cc_number,
                expiry: customerData.cc_expiry,
                status: customerData.cc_status,
            },
        };
        bookingObj.reservation.pickup_location = customerData.address
            ? customerData.address
            : "";
        bookingObj.payment_info.payment_type = customerData.pay_type_id
            ? customerData.pay_type_id
            : "";

        const [cc_month, cc_year] = customerData.pay_type_id === 3
                ?  customerData.cc_expiry.split("-")
                : []; // Split the string
        bookingObj.payment_info.cc_info =
            customerData.pay_type_id === 3
                ? {
                      card_type: customerData.card_type,
                      cc_number: customerData.cc_number,
                      cc_expiry_month: cc_month,
                      cc_expiry_year: cc_year,
                      csv_code: customerData.csv_code,
                      cc_status: customerData.cc_status,
                      billing_zip_code: customerData.billing_zip_code,
                      card_holder_name: customerData.card_holder_name,
                      billing_address: customerData.billing_address,
                      billing_phone_number: customerData.billing_phone_number,
                      is_address_same: customerData.is_address_same,
                  }
                : bookingObj.payment_info.cc_info;

        return bookingObj;
    }
    static setReservationFromRaw(rawData) {
        let bookingObj = cloneDeep(default_booking);

        bookingObj.service_settings.service_type_id = rawData.service_type_id;
        bookingObj.service_settings.is_edit = true;
        //bookingObj.service_settings.is_round_trip = rawData.round_trip;
        bookingObj.service_settings.customer_notes = rawData.customer_notes;
        bookingObj.customer = {
            id: rawData.customer.id,
            first_name: rawData.customer.first_name,
            last_name: rawData.customer.last_name,
            mobile_number: rawData.customer.mobile_number,
            phone_number: rawData.customer.phone_number,
            email: rawData.customer.email,
            cc: {
                type: rawData.customer.card_type,
                number: rawData.customer.cc_number,
                expiry: rawData.customer.cc_expiry,
                status: rawData.customer.cc_status,
            },
        };
        bookingObj.reservation.customer_picked_date_time = rawData.milage_fare_detail.timely_charges ? true : false
        bookingObj.payment_info.payment_type = rawData.customer.pay_type_id
            ? rawData.customer.pay_type_id
            : "";

        bookingObj.reservation.id = rawData.id;

        bookingObj.reservation.pickup_date = rawData.pick_up_date;
        bookingObj.reservation.pickup_time = rawData.pick_up_time;
        bookingObj.reservation.pickup_location = rawData.pick_up_location;
        bookingObj.reservation.dropoff_location = rawData.drop_off_location;

        bookingObj.reservation.flight_info = {
            flight_type:
                rawData.airport_ride === "to_airport" ? "departure" : "arrival",
            flight_date:
                rawData.airport_ride === "to_airport"
                    ? rawData.departure_pick_up_date
                    : rawData.pick_up_date,
            flight_time:
                rawData.airport_ride === "to_airport"
                    ? rawData.departure_pick_up_time
                    : rawData.pick_up_time,
            flight_airline: rawData.selected_airline,
            flight_number: rawData.flight_number,
            flight_destination: rawData.selected_flight_destination,
            selected_airport:
                rawData.service_type_id !== 1
                    ? {}
                    : {
                          name:
                              rawData.airport_ride === "to_airport"
                                  ? rawData.drop_off_location.description
                                  : rawData.pick_up_location.description,
                          airport_location:
                              rawData.airport_ride === "to_airport"
                                  ? rawData.drop_off_location
                                  : rawData.pick_up_location,
                      },
        };
        bookingObj.reservation.time_duration_text = rawData.time_duration_text;
        bookingObj.reservation.distance = rawData.distance;
        bookingObj.reservation.time_duration_hours =
            rawData.time_duration_hours;
        bookingObj.reservation.ride_type = rawData.ride_type;
        bookingObj.reservation.additional_stops = rawData.additional_stops
            ? rawData.additional_stops
            : [];
        bookingObj.reservation.travellers = rawData.travellers_types;
        bookingObj.reservation.luggage = rawData.luggage;
        bookingObj.reservation.drop_off_hours = rawData.drop_off_hours;
        bookingObj.reservation.is_on_call = rawData.is_on_call
            ? rawData.is_on_call
            : false;

        bookingObj.reservation.coupon = rawData.coupon
            ? rawData.coupon
            : bookingObj.reservation.coupon;
        bookingObj.reservation.admin_discount = rawData.admin_discount;
        bookingObj.reservation.driver_gratuity = rawData.driver_gratuity;
        //bookingObj.reservation.reservationCity = rawData.city_id;

        //bookingObj.reservation.selected_vehicle = rawData.selected_vehicle;
        bookingObj.reservation.selected_vehicle = {
            id: rawData.vehicle_type_id,
            ride_type: rawData.ride_type,
            passengers_capacity: rawData.selected_vehicle?.passengers_capacity,
            milage_fare_detail: rawData.milage_fare_detail,
            hourly_fare_detail: rawData.hourly_fare_detail,
        };
        bookingObj.reservation.timeLimitError = true;
        bookingObj.reservation.reservationCity = {
            id: rawData.city_id,
        };
        const [cc_month, cc_year] = rawData.customer.pay_type_id === 3?rawData.customer.cc_expiry.split("-"):[]; // Split the string
        bookingObj.payment_info.cc_info =
            rawData.customer.pay_type_id === 3
                ? {
                      card_type: rawData.customer.card_type,
                      cc_number: rawData.customer.cc_number,
                      cc_expiry_month: cc_month,
                      cc_expiry_year: cc_year,
                      csv_code: rawData.customer.csv_code,
                      cc_status: rawData.customer.cc_status,
                      billing_zip_code: rawData.customer.billing_zip_code,
                      card_holder_name: rawData.customer.card_holder_name,
                      billing_address: rawData.customer.billing_address,
                      billing_phone_number: rawData.customer.billing_phone_number,
                      is_address_same: rawData.customer.is_address_same,
                  }
                : bookingObj.payment_info.cc_info;

        return bookingObj;
    }
    getReservationFromRaw(rawData) {
        let bookingObj = cloneDeep(default_booking);
        bookingObj.service_settings.service_type_id = rawData.service_type_id;
        //bookingObj.service_settings.is_round_trip = rawData.round_trip;
        bookingObj.service_settings.customer_notes = rawData.customer_notes;
        bookingObj.customer = rawData.customer;
        bookingObj.reservation.pickup_date = rawData.pick_up_date;
        bookingObj.reservation.pickup_time = rawData.pick_up_time;
        bookingObj.reservation.pickup_location = rawData.pick_up_location;
        bookingObj.reservation.dropoff_location = rawData.drop_off_location;

        bookingObj.reservation.flight_info = {
            flight_type:
                rawData.airport_ride === "to_airport" ? "departure" : "arrival",
            selected_airport:
                rawData.airport_ride === "to_airport"
                    ? rawData.drop_off_location.address
                    : rawData.pick_up_location.address,
            flight_date:
                rawData.airport_ride === "to_airport"
                    ? rawData.departure_pick_up_date
                    : rawData.pick_up_date,
            flight_time:
                rawData.airport_ride === "to_airport"
                    ? rawData.departure_pick_up_time
                    : rawData.pick_up_time,
            flight_airline: rawData.selected_airline,
            flight_number: rawData.flight_number,
            flight_destination: rawData.selected_flight_destination,
        };
        bookingObj.reservation.time_duration_text = rawData.time_duration_text;
        bookingObj.reservation.distance = rawData.distance;
        bookingObj.reservation.time_duration_hours =
            rawData.time_duration_hours;
        bookingObj.reservation.ride_type = rawData.ride_type;
        bookingObj.reservation.additional_stops = rawData.additional_stops
            ? rawData.additional_stops
            : [];
        bookingObj.reservation.travellers = rawData.travellers_types;
        bookingObj.reservation.luggage = rawData.luggage;
        bookingObj.reservation.drop_off_hours = rawData.drop_off_hours;
        bookingObj.reservation.is_on_call = rawData.is_on_call
            ? rawData.is_on_call
            : false;
        bookingObj.reservation.coupon = rawData.coupon
            ? rawData.coupon
            : bookingObj.reservation.coupon;
        bookingObj.reservation.admin_discount = rawData.admin_discount;
        bookingObj.reservation.driver_gratuity = rawData.driver_gratuity;
        bookingObj.reservation.reservationCity = rawData.city_id;

        //bookingObj.reservation.selected_vehicle = rawData.selected_vehicle;
        bookingObj.reservation.selected_vehicle = {
            id: rawData.vehicle_type_id,
            ride_type: rawData.ride_type,
            passengers_capacity: rawData.selected_vehicle?.passengers_capacity,
            milage_fare_detail: rawData.milage_fare_detail,
            hourly_fare_detail: rawData.hourly_fare_detail,
        };

        return new ReservationClass(
            bookingObj.reservation,
            bookingObj.service_settings
        );
    }
    getDepartureTimeLimit(objVehicle) {
        let objVehicleFareDetail = {};
        let addMinutesToDeparturePickup = 0;
        if (objVehicle.ride_type === "hourly") {
            objVehicleFareDetail = objVehicle.hourly_fare_detail;
        } else if (objVehicle.ride_type === "zip_code") {
            objVehicleFareDetail = objVehicle.milage_fare_detail;
        } else {
            objVehicleFareDetail = objVehicle.milage_fare_detail;
        }
        //if (this.service_settings.service_type_id === 1)
        //    addMinutesToDeparturePickup = this.reservation.flight_info.flight_type === "departure" ? Number(this.reservation.time_duration_hours) * 60 : 0;
        addMinutesToDeparturePickup = 0;

        const totalMinutes =
            Number(objVehicleFareDetail.departure_time_limit.pickup_hours) *
                60 +
            Number(objVehicleFareDetail.departure_time_limit.pickup_minutes) +
            addMinutesToDeparturePickup;

        // Round the total time to the nearest multiple of 5
        const roundedMinutes = Math.round(totalMinutes / 15) * 15;

        return {
            hours: objVehicleFareDetail.departure_time_limit.pickup_hours,
            minutes: objVehicleFareDetail.departure_time_limit.pickup_minutes,
            total_minutes: roundedMinutes,
        };
    }
    sub_total_fare(objVehicle) {
        let objVehicleFareDetail = {},
            baseFare = 0;
        if (objVehicle.ride_type === "hourly") {
            objVehicleFareDetail = objVehicle.hourly_fare_detail;
        } else if (objVehicle.ride_type === "zip_code") {
            objVehicleFareDetail = objVehicle.milage_fare_detail;
        } else {
            objVehicleFareDetail = objVehicle.milage_fare_detail;
        }

        const additionalCharges = this.sumAdditionalCharges(
            objVehicleFareDetail.additional_charges
        );
        const timeBaseCharges = objVehicleFareDetail.timely_charges.reduce(
            (acc, charge) => acc + charge.amount,
            0
        );
        const discounts = objVehicleFareDetail.discounts.reduce(
            (acc, charge) => acc + charge.amount,
            0
        );
        return (
            objVehicleFareDetail.base_fare +
            (additionalCharges + timeBaseCharges) -
            discounts
        );
    }

    total_fare(objVehicle) {
        return (this.reservation.total_fare =
            this.sub_total_fare(objVehicle) * this.totalVehiclesRequired(objVehicle) +
            Number(this.reservation.driver_gratuity || 0) -
            Number(this.reservation.admin_discount || 0) -
            (Number(this.reservation.coupon.coupon_price)
                ? Number(this.reservation.coupon.coupon_price)
                : 0));
    }

    getAdditionalCharges(charges) {
        //alert(this.service_settings.service_type_id);
        let filtered_charges = charges
            .filter(
                (charge) =>
                    !(charge.id === 10 && this.reservation.is_on_call === false)
            )
            .filter(
                (charge) =>
                    !(
                        charge.id === 4 &&
                        this.service_settings.service_type_id !== 1
                    )
            )
            .filter(
                (charge) =>
                    !(
                        charge.id === 4 &&
                        this.reservation.flight_info.flight_type !== "arrival"
                    )
            );

        return filtered_charges;
    }
    sumAdditionalCharges(charges) {
        let sum = 0;
        this.getAdditionalCharges(charges).forEach((charge) => {
            sum += charge.amount;
        });
        return sum;
    }

    totalTravellers() {
        // Check if travellers is a valid array and return the sum of counts
        if (this.reservation.travellers && Array.isArray(this.reservation.travellers)) {
            return this.reservation.travellers.reduce(
                (total, obj) => total + (parseInt(obj.count) || 0),
                0
            );
        }
        return 0;
    }
    totalAdditionStops() {
        // this.reservation.additional_stops = (this.reservation.additional_stops.filter(obj => obj.address !== undefined));
        return this.reservation.additional_stops.length;
    }

    totalVehicles(vehicle) {
        return Math.ceil(
            this.totalTravellers() / parseInt(vehicle.passengers_capacity)
        );
    }

    totalVehiclesForLuggage(vehicle) {
        if (!this.reservation.luggage || this.reservation.luggage <= 0) {
            return 1; // At least 1 vehicle is needed
        }
        
        const luggageCount = parseInt(this.reservation.luggage);
        const vehicleLuggageCapacity = parseInt(vehicle.luggage_capacity);
        
        // Calculate vehicles needed based on luggage capacity
        return Math.ceil(luggageCount / vehicleLuggageCapacity);
    }

    totalVehiclesRequired(vehicle) {
        const travellersVehicles = this.totalVehicles(vehicle);
        const luggageVehicles = this.totalVehiclesForLuggage(vehicle);
        
        // Return the maximum of both requirements
        return Math.max(travellersVehicles, luggageVehicles);
    }

    getPickupLocation() {
        return this.reservation.pickup_location.address;
    }
    getPickupDateTime() {
        return (
            this.reservation.pickup_time + " " + this.reservation.pickup_date
        );
    }
    getPickupDateTimeMoment() {
        return moment(
            this.reservation.pickup_date + " " + this.reservation.pickup_time,
            "YYYY-MM-DD HH:mm:ss"
        );
    }
    getFlightDateTimeMoment() {
        return moment(
            this.reservation.flight_info.flight_date +
                " " +
                this.reservation.flight_info.flight_time,
            "YYYY-MM-DD HH:mm:ss"
        );
    }
    getDropOffLocation() {
        return this.reservation.dropoff_location.address;
    }
    getDropOffLocation() {
        return this.reservation.dropoff_location.address;
    }
    getDistance() {
        return this.reservation.distance;
    }
    getDDropOfHours() {
        return this.reservation.drop_off_hours;
    }
}

export const default_booking = {
    service_settings: {
        reservation_id: 0,
        service_type_id: 0,
        is_round_trip: false,
        customer_notes: "",
        is_admin: false,
        fareQuote: false,
        is_edit: false,
        check_proceed: false
    },
    customer: {
        id: 0,
        first_name: "",
        last_name: "",
        mobile_number: "",
        phone_number: "",
        email: "",
    },
    reservation: {
        id: 0,
        customer_picked_date_time: false,
        pickup_date: "",
        pickup_time: "",
        pickup_location: {
            address: "",
        },
        dropoff_location: {
            address: "",
        },
        flight_info: {
            flight_type: "departure",
            selected_airport: "",
            flight_date: "",
            flight_time: "",
            flight_airline: "",
            flight_number: "",
            flight_destination: "",
        },
        time_duration_text: "",
        distance: 0,
        time_duration_hours: "0",
        ride_type: "mileage",
        additional_stops: [],
        travellers: [
            { id: 1, name: "Adult", count: 0 },
            { id: 2, name: "Child", count: 0 },
            { id: 3, name: "Infant", count: 0 },
            { id: 4, name: "Military", count: 0 },
            { id: 5, name: "Students", count: 0 },
            { id: 6, name: "Senior Citizen", count: 0 }
        ],
        luggage: "",
        drop_off_hours: 1,
        scheduled: {
            ride_type: "",
        },
        is_on_call: false,
        selected_vehicle: {},
        admin_discount: 0,
        driver_gratuity: 0,
        total_fare: 0,
        timeLimitError: "",
        reservationCity: "",
        coupon: {
            id: "",
            coupon_code: "",
            coupon_price: "",
            coupon_type: "",
            expiry_date: "",
        },
    },
    reservation_2: {
        id: 0,
        customer_picked_date_time: false,
        pickup_date: "",
        pickup_time: "",
        pickup_location: {
            address: "",
        },
        dropoff_location: {
            address: "",
        },
        flight_info: {
            flight_type: "departure",
            selected_airport: "",
            flight_date: "",
            flight_time: "",
            flight_airline: "",
            flight_number: "",
            flight_destination: "",
        },
        time_duration_text: "",
        distance: 0,
        time_duration_hours: "0",
        ride_type: "mileage",
        additional_stops: [],
        travellers: [
            { id: 1, name: "Adult", count: 0 },
            { id: 2, name: "Child", count: 0 },
            { id: 3, name: "Infant", count: 0 },
            { id: 4, name: "Military", count: 0 },
            { id: 5, name: "Students", count: 0 },
            { id: 6, name: "Senior Citizen", count: 0 }
        ],
        luggage: "",
        drop_off_hours: 1,
        scheduled: {
            ride_type: "",
        },
        is_on_call: false,
        selected_vehicle: {},
        admin_discount: 0,
        driver_gratuity: 0,
        total_fare: 0,
        timeLimitError: "",
        reservationCity: "",
        coupon: {
            id: "",
            coupon_code: "",
            coupon_price: "",
            coupon_type: "",
            expiry_date: "",
        },
    },
    payment_info: {
        payment_type: 1,
        use_existing_cc: false,
        cc_info: {
            card_type: "",
            cc_number: "",
            cc_expiry_month: "",
            cc_expiry_year: "",
            csv_code: "",
            cc_status: "",
            billing_zip_code: "",
            card_holder_name: "",
            billing_address: "",
            billing_phone_number: "",
            is_address_same: "",
        },
        direct_account: {
            account_id: "",
            account_name: "",
            account_number: "",
            account_verified: false,
        },
    },
    coupon: {
        id: "",
        coupon_code: "",
        coupon_price: "",
        coupon_type: "",
        expiry_date: "",
    },
    customer_notes: "",
    completed: false,
};

export const default_test_booking = {
    service_settings: {
        service_type_id: 1,
        is_round_trip: true,
        customer_notes: "",
    },
    customer: {
        id: 0,
        first_name: "Iftikhar",
        last_name: "Ali",
        mobile_number: "03216017941",
        phone_number: "048768719",
        email: "mriftikhar@gmail.com",
        verified_linked_cc: {
            cc_type: "",
            cc_ending_number: "",
            cc_expiry: "",
        },
    },
    reservation: {
        service_type_id: 1,
        pickup_date: "2024-01-10",
        pickup_time: "12:00",
        pickup_location: {
            address: "8511 Oxford Ave, Raytown, MO 64138, USA",
            latitude: 38.9677946,
            longitude: -94.4726736,
            country: "United States",
            state: "Missouri",
            county: "Jackson County",
            city: "Raytown",
            postcode: "64138",
        },
        dropoff_location: {
            city: "Kansas City",
            state: "Missouri",
            county: "Platte County",
            address: "1 International Square, Kansas City, MO 64153, USA",
            country: "United States",
            latitude: 39.2976967,
            postcode: "64153",
            longitude: -94.7166434,
        },
        flight_info: {
            flight_type: "departure",
            selected_airport: {
                id: 1,
                name: "1 International Square, Kansas City, MO 64153, USA",
                airport_location: {
                    city: "Kansas City",
                    state: "Missouri",
                    county: "Platte County",
                    address:
                        "1 International Square, Kansas City, MO 64153, USA",
                    country: "United States",
                    latitude: 39.2976967,
                    postcode: "64153",
                    longitude: -94.7166434,
                },
            },
            flight_date: "2024-01-10",
            flight_time: "12:00",
            flight_airline: "AMERICAN AIRLINE",
            flight_number: "545645646",
            flight_destination: "Albuquerque",
        },
        time_duration_text: "39 minutes",
        distance: 34,
        time_duration_hours: "0.65",
        ride_type: "mileage",
        additional_stops: [],
        travellers: [
            { id: 1, name: "Adult", count: 0 },
            { id: 2, name: "Child", count: 0 },
            { id: 3, name: "Infant", count: 0 },
            { id: 4, name: "Military", count: 0 },
            { id: 5, name: "Students", count: 0 },
            { id: 6, name: "Senior Citizen", count: 0 }
        ],
        luggage: 0,
        drop_off_hours: 1,
        scheduled: {
            ride_type: "",
        },
        is_on_call: false,
        selected_vehicle: {},
        admin_discount: 0,
        driver_gratuity: 0,
        reservationCity: "",
        coupon: {
            id: "",
            coupon_code: "",
            coupon_price: "",
            coupon_type: "",
            expiry_date: "",
        },
    },
    reservation_2: {
        service_type_id: 1,
        pickup_date: "2024-01-11",
        pickup_time: "12:00",
        dropoff_location: {
            address: "8511 Oxford Ave, Raytown, MO 64138, USA",
            latitude: 38.9677946,
            longitude: -94.4726736,
            country: "United States",
            state: "Missouri",
            county: "Jackson County",
            city: "Raytown",
            postcode: "64138",
        },
        pickup_location: {
            city: "Kansas City",
            state: "Missouri",
            county: "Platte County",
            address: "1 International Square, Kansas City, MO 64153, USA",
            country: "United States",
            latitude: 39.2976967,
            postcode: "64153",
            longitude: -94.7166434,
        },
        flight_info: {
            flight_type: "arrival",
            selected_airport: {
                id: 1,
                name: "1 International Square, Kansas City, MO 64153, USA",
                airport_location: {
                    city: "Kansas City",
                    state: "Missouri",
                    county: "Platte County",
                    address:
                        "1 International Square, Kansas City, MO 64153, USA",
                    country: "United States",
                    latitude: 39.2976967,
                    postcode: "64153",
                    longitude: -94.7166434,
                },
            },
            flight_date: "2024-01-11",
            flight_time: "12:00",
            flight_airline: "AMERICAN AIRLINE",
            flight_number: "545645646",
            flight_destination: "Albuquerque",
        },
        time_duration_text: "39 minutes",
        distance: 34,
        time_duration_hours: "0.65",
        ride_type: "mileage",
        additional_stops: [],
        travellers: [
            { id: 1, name: "Adult", count: 0 },
            { id: 2, name: "Child", count: 0 },
            { id: 3, name: "Infant", count: 0 },
            { id: 4, name: "Military", count: 0 },
            { id: 5, name: "Students", count: 0 },
            { id: 6, name: "Senior Citizen", count: 0 }
        ],
        luggage: 0,
        drop_off_hours: 1,
        scheduled: {
            ride_type: "",
        },
        is_on_call: false,
        selected_vehicle: {},
        admin_discount: 0,
        driver_gratuity: 0,
        reservationCity: "",
        coupon: {
            id: "",
            coupon_code: "",
            coupon_price: "",
            coupon_type: "",
            expiry_date: "",
        },
    },
    payment_info: {
        payment_type: 1,
        cc_info: {
            card_type: "",
            cc_number: "",
            cc_expiry_month: "",
            cc_expiry_year: "",
            csv_code: "",
            cc_status: "",
            billing_zip_code: "",
            card_holder_name: "",
            billing_address: "",
            billing_phone_number: "",
            is_address_same: "",
        },
        direct_account: {
            account_id: 1,
            account_name: "",
            account_number: "",
        },
    },
    coupon: {
        id: "",
        coupon_code: "",
        coupon_price: "",
        coupon_type: "",
        expiry_date: "",
    },
    customer_notes: "",
    completed: false,
};
