<?php
namespace App\Services;

use Illuminate\Support\Facades\Mail;
use App\Models\Admin\ActivityLog;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Log;


class EmailService
{
    /**
     * Send an email using a specified Mailable class.
     *
     * @param string $email
     * @param \Illuminate\Mail\Mailable $mailable
     * @return bool
     */
    public function sendEmail($email, $mailable, $subject, $reservationIds = [],$customerId)
    {
        try {
            // Send the email
            Mail::to($email)->send($mailable);

            // Store the log after email is sent
            $this->storeLog(true, $reservationIds, $subject,$customerId);

            return true;
        } catch (\Exception $exception) {
            return false;
        }
    }

    /**
     * Queue an email to be sent later using a specified Mailable class.
     *
     * @param string $email
     * @param \Illuminate\Mail\Mailable $mailable
     * @return bool
     */
    public function queueEmail($email, $mailable)
    {
        try {
            // Queue the email
            Mail::to($email)->queue($mailable);

            // Return true if email is queued successfully
            return true;
        } catch (\Exception $exception) {
            // Log the error and return false
            \Log::error('Failed to queue email: ' . $exception->getMessage());
            return false;
        }
    }

    public function storeLog($emailSent, $reservationIds, $subject,$customerId)
    {
        if ($emailSent) {
            foreach ($reservationIds as $reservationId) {
                $route = ActivityLog::where('item_id', $reservationId)->first();
                $latestRoute = $route ? $route->route : '';
                $ipAddress = request()->ip();
                $user = Auth::user();
                $containsDriver = strpos($latestRoute, 'driver') !== false && strpos($latestRoute, 'admin') === false;
                $containsCustomer = strpos($latestRoute, 'with-user') !== false;
                $containsGuestCustomer = strpos($latestRoute, 'with-guest-user') !== false;

                $roleId = determineRoleId($containsDriver, $containsCustomer, $user);
                $userId = ($containsGuestCustomer && Auth::id() === null) ? null : ($containsCustomer && isset($customerId) ? $customerId : ($user->id ?? null));

                // Remove newline characters from the message
                $message = str_replace(["\r", "\n"], '', $subject);
                $jsonEncodedMessage = json_encode($message);
                Log::info([$userId,$roleId,$ipAddress,$customerId,$reservationId,$subject,$jsonEncodedMessage]);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    Log::error('JSON encoding error: ' . json_last_error_msg());
                } else {
                    // Store the activity log in the database
                    ActivityLog::create([
                        'user_id' => $userId,
                        'role_id' => $roleId,
                        'ip_address' => $ipAddress,
                        'customer_id' => $customerId ?? null,
                        'item_id' => $reservationId,
                        'table_name' => 'reservations',
                        'activity' => 'email',
                        'route' => URL::current(),
                        'activity_description' => $subject,
                        'activity_detail' => $jsonEncodedMessage,
                    ]);
                }
            }
        }
    }
}
