<?php

namespace App\Http\Controllers\Admin\Reservations;

use stdClass;

use Carbon\Carbon;
use Google\Client;
use Google_Client;
use App\Mail\SendEmail;
use Google\Service\Gmail;
use Google_Service_Gmail;
use App\Models\Admin\Coupon;
use App\Models\Admin\Driver;
use Illuminate\Http\Request;
use App\Models\Admin\Airline;
use App\Models\Admin\Airport;
use App\Models\Admin\PayType;
use App\Models\Admin\Vehicle;
use App\Models\Admin\Customer;
use App\Mail\NewUserWelcomeMail;
use App\Models\Admin\BlockDriver;
use App\Models\Admin\Reservation;
use Google\Service\Gmail\Message;
use Google_Service_Gmail_Message;
use Illuminate\Support\Facades\DB;
use PHPMailer\PHPMailer\PHPMailer;
use App\Models\Admin\DirectAccount;
use App\Models\Admin\GuestCustomer;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\View;
use App\Models\Admin\FlightDestination;
use Illuminate\Support\Facades\Validator;
use Google\Service\ServiceManagement\Service;

class ReservationsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('admin.reservations.index');
    }


    public function reservationIndex($customer = null, $customerType = null)
    {
        return view('frontend.reservations.reservations', [
            'customer' => $customer,
            'customer_type' => $customerType,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, $customerId, $customerType, $payTypeId, $isRoundTrip)
    {
        try {
            // Common reservation fields
            $reservation_obj = new Reservation();

            if (!$isRoundTrip) {
                $reservationData = $request->reservation_data;
            } else if ($isRoundTrip) {
                $reservationData = $request->round_trip_reservation_data;
            }

            $reservation_obj->user_id = Auth::id() ? Auth::id() : null;
            $reservation_obj->city_id = $reservationData['city_id'];
            $reservation_obj->service_type_id = $reservationData['service_type_id'];
            $reservation_obj->ip_address = $request->ip();
            $reservation_obj->customer_id = $customerId;
            $reservation_obj->customer_type = $customerType;
            $reservation_obj->site_id = $request->site_id ? $request->site_id : 1;

            if ($reservationData['service_type_id'] !== 4) {
                $reservation_obj->vehicle_type_id = $reservationData['selected_service']['id'];
                $reservation_obj->departure_pick_up_date = $reservationData['departure_pick_up_date'];
                $reservation_obj->departure_pick_up_time = $reservationData['departure_pick_up_time'];
                $reservation_obj->pick_up_date = $reservationData['pick_up_date'];
                $reservation_obj->pick_up_time = $reservationData['pick_up_time'];
                $reservation_obj->travellers = $reservationData['travellers'];
                $reservation_obj->luggage = $reservationData['luggage'];
                $reservation_obj->pick_up_location = $reservationData['pick_up_location'];
                $reservation_obj->drop_off_location = $reservationData['drop_off_location'];
                $reservation_obj->ride_type = $reservationData['selected_service']['ride_type'];
            }

            // Travellers types if exist (Not for Schedule service)
            $reservation_obj->travellers_types = $reservationData['travellers_types'] !== [] ? $reservationData['travellers_types'] : null;
            $reservation_obj->selected_airline = $reservationData['selected_airline'] !== [] ? $reservationData['selected_airline'] : null;
            $reservation_obj->selected_flight_destination = $reservationData['selected_flight_destination'] !== [] ? $reservationData['selected_flight_destination'] : null;
            $reservation_obj->flight_number = $reservationData['flight_number'] !== [] ? $reservationData['flight_number'] : null;

            //Airport ride for airport services
            $reservation_obj->airport_ride = $reservationData['airport_ride'] ? $reservationData['airport_ride'] : null;

            //Additional stops, distances and time values if exist
            $reservation_obj->additional_stops = count($reservationData['additional_stops']) > 0 ? $reservationData['additional_stops'] : null;

            $reservation_obj->distance = $reservationData['distance'] ? $reservationData['distance'] : null;
            $reservation_obj->time_duration_text = $reservationData['time_duration_text'] ? $reservationData['time_duration_text'] : null;
            // $reservation_obj->time_duration_hours = $reservationData['time_duration_hours'] ? $reservationData['time_duration_hours'] : null;

            $reservation_obj->vehicles_count = $reservationData['vehicles_count'] ? $reservationData['vehicles_count'] : null;

            // Number of Hours for 'Airport(hourly)' and 'Hourly' Services
            $reservation_obj->drop_off_hours = $reservationData['drop_off_hours'] !== '' ? $reservationData['drop_off_hours'] : null;

            //For Airport hourly  service only
            if ($reservationData['service_type_id'] != 4 && $reservationData['service_type_id'] == 1 && $reservationData['selected_service']) {
                if ($reservationData['selected_service']['ride_type'] == 'hourly') {
                    $reservation_obj->drop_off_hours = $reservationData['selected_service']['drop_off_hours'];
                }
            }

            if ($reservationData['service_type_id'] !== 4 && $reservationData['selected_service']) {

                if ($isRoundTrip) {
                    $reservation_obj->price = $request->round_trip_total_fare;
                    $reservation_obj->admin_discount = $request->round_admin_discount;
                    $reservation_obj->is_on_call = $request->is_round_on_call;
                } else if (!$isRoundTrip) {
                    $reservation_obj->price = $request->total_fare;
                    $reservation_obj->admin_discount = $request->admin_discount;
                    $reservation_obj->is_on_call = $request->is_on_call;
                }

                if ($reservationData['selected_service']['ride_type'] == 'millage') {
                    // If service selected for millage ride type
                    $reservation_obj->additional_millage_charges = $reservationData['selected_service']['millage_service']['additional_millage_charges'];
                    $reservation_obj->timely_millage_charges = $reservationData['selected_service']['millage_service']['timely_millage_charges'];
                    $reservation_obj->traveller_types_charges = $reservationData['selected_service']['millage_service']['traveller_types_charges'];
                    // $reservation_obj->price = $reservationData['selected_service']['millage_service']['est_with_add_fare'];
                } else if ($reservationData['selected_service']['ride_type'] == 'hourly') {
                    // If service selected for hourly ride type
                    $reservation_obj->additional_millage_charges = $reservationData['selected_service']['hourly_service']['additional_hourly_charges'];
                    $reservation_obj->timely_millage_charges = $reservationData['selected_service']['hourly_service']['timely_hourly_charges'];
                    $reservation_obj->traveller_types_charges = $reservationData['selected_service']['hourly_service']['traveller_types_charges'];
                    // $reservation_obj->price = $reservationData['selected_service']['hourly_service']['est_with_add_fare'];
                }
            }

            if ($reservationData['service_type_id'] == 4) {
                if (!$isRoundTrip) {
                    $this->makeScheduleReservations($reservation_obj, $request->schedule_service);
                } else {
                    $this->makeScheduleReservations($reservation_obj, $request->round_schedule_service);
                }
            }

            $reservation_obj->pay_type_id = $payTypeId;
            $reservation_obj->status = 'new_reservation';

            $reservation_obj->save();

            return $reservation_obj; // Return the created reservation object

        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    public function makeScheduleReservations($reservation_obj, $schedule_data)
    {

        // dd($schedule_data);

        $reservation_obj->shuttle_name = $schedule_data['shuttle_name'];
        $reservation_obj->schedule_service_type = $schedule_data['ride_type'];
        $reservation_obj->vehicle_type_id = $schedule_data['vehicle_type_id'];
        $reservation_obj->departure_pick_up_date = $schedule_data['departure_pick_up_date'] ? $schedule_data['departure_pick_up_date'] : null;
        $reservation_obj->departure_pick_up_time = $schedule_data['departure_pick_up_time'] ? $schedule_data['departure_pick_up_time'] : null;
        $reservation_obj->pick_up_date = $schedule_data['pick_up_date'] ?? null;

        $reservation_obj->pick_up_time = $schedule_data['pick_up_time'] ? $schedule_data['pick_up_time'] : null;
        $reservation_obj->drop_off_time = $schedule_data['drop_off_time'];

        $reservation_obj->travellers = $schedule_data['scheduleServiceAdditionInformation']['travellers'];
        $reservation_obj->luggage = $schedule_data['scheduleServiceAdditionInformation']['carry_bags_capacity'];
        $reservation_obj->check_in_bags = $schedule_data['scheduleServiceAdditionInformation']['check_in_bags_capacity'];
        $reservation_obj->pick_up_location = $schedule_data['pick_up_location'];
        $reservation_obj->drop_off_location = $schedule_data['drop_off_location'];
        $reservation_obj->selected_airline = $schedule_data['selected_airline'] ? $schedule_data['selected_airline'] : null;
        $reservation_obj->selected_flight_destination = $schedule_data['selected_flight_destination'] ? $schedule_data['selected_flight_destination'] : null;
        $reservation_obj->flight_number = $schedule_data['flight_number'] ? $schedule_data['flight_number'] : null;

        if (isset($schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_location']) && isset($schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_location']['address'])) {
            $reservation_obj->home_pick_up_location = $schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_location']['address'];
        }
        if (isset($schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_location']) && isset($schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_location']['address'])) {
            $reservation_obj->home_drop_off_location = $schedule_data['scheduleServiceAdditionInformation']['schedule_drop_off_location']['address'];
        }

        if ($schedule_data['ride_type'] == 'airport' && $schedule_data['airport_ride_type'] == 'to_airport') {
            $reservation_obj->airport_ride = $schedule_data['airport_ride_type'];
            if (isset($schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address']) && isset($schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address']['address'])) {
                $reservation_obj->home_pick_up_location = $schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address'];
            }
        }
        if ($schedule_data['ride_type'] == 'airport' && $schedule_data['airport_ride_type'] == 'from_airport') {
            $reservation_obj->airport_ride = $schedule_data['airport_ride_type'];
            if (isset($schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address']) && isset($schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address']['address'])) {
                $reservation_obj->home_drop_off_location = $schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address'] !== '' ? $schedule_data['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address'] : '';
            }
        }

        $reservation_obj->vehicle_capacity = $schedule_data['vehicle_capacity'] ? $schedule_data['vehicle_capacity'] : null;
        $reservation_obj->distance = $schedule_data['distance'] ? $schedule_data['distance'] : null;
        $reservation_obj->time_duration_text = $schedule_data['time_duration_text'] ? $schedule_data['time_duration_text'] : null;
        $reservation_obj->additional_millage_charges = $schedule_data['additional_millage_charges'];
        $reservation_obj->timely_millage_charges = $schedule_data['timely_millage_charges'];
        $reservation_obj->price = $schedule_data['fare_with_add_charges'];
        $reservation_obj->fare_per_person = $schedule_data['fare_per_person'];
        $reservation_obj->fare_with_add_charges = $schedule_data['fare_with_add_charges'];

        // dd($reservation_obj);
    }

    public function getReservationData()
    {
        try {
            $obj = new stdClass();

            $data = Reservation::latest()->first();

            $obj->data = $data;

            return response()->json($obj, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function getRegisteredCustomer(Request $request)
    {
        try {

            $obj = new stdClass();

            $customer = Customer::where('email', $request->email)->first();

            if ($customer && Hash::check($request->password, $customer->password)) {
                $cc_expiry = $customer->cc_expiry;
                if (!empty($cc_expiry)) {
                    $cc_expiry_arr = explode('-', $cc_expiry); // Split the cc_expiry string into an array

                    $cc_expiry_month = $cc_expiry_arr[0];
                    $cc_expiry_year = $cc_expiry_arr[1];

                    $customer->cc_expiry_month = $cc_expiry_month;
                    $customer->cc_expiry_year = $cc_expiry_year;
                }
            } else {
                $customer = null;
            }

            $customer ? $customer->password = '' : null;

            $obj->customer = $customer;

            return response()->json($obj, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function getAllReservations()
    {
        try {
            $obj = new stdClass();
            $obj->data = [];

            $reservations = Reservation::all();

            foreach ($reservations as $reservation) {
                if ($reservation->customer_type == 'Guest' || $reservation->customer_type == 'guest') {
                    $reservation->customer = GuestCustomer::find($reservation->customer_id);
                } elseif ($reservation->customer_type == 'registered') {
                    $reservation->customer = Customer::find($reservation->customer_id);
                }

                $obj->data[] = $reservation;
            }

            return response()->json($obj, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function getTodayMadeReservations()
    {
        try {

            $reservations = Reservation::whereDate('created_at', '=', Carbon::today()->toDateString())->get();

            foreach ($reservations as $reservation) {
                if ($reservation->customer_type == 'Guest' || $reservation->customer_type == 'guest') {
                    $reservation->customer = GuestCustomer::find($reservation->customer_id);
                } else if ($reservation->customer_type == 'registered' || $reservation->customer_type == 'Registered') {
                    $reservation->customer = Customer::find($reservation->customer_id);
                }
            }
            return response()->json($reservations, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function getTodayReservations()
    {
        try {
            $reservations = Reservation::whereDate('pick_up_date', '=', Carbon::today()->toDateString())->get();

            foreach ($reservations as $reservation) {
                if ($reservation->customer_type == 'Guest' || $reservation->customer_type == 'guest') {
                    $reservation->customer = GuestCustomer::find($reservation->customer_id);
                } else if ($reservation->customer_type == 'registered' || $reservation->customer_type == 'Registered') {
                    $reservation->customer = Customer::find($reservation->customer_id);
                }
            }
            return response()->json($reservations, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function getTomorrowReservations()
    {
        try {
            // dd(Carbon::tomorrow()->toDateString());
            $reservations = Reservation::whereDate('pick_up_date', '=', Carbon::tomorrow()->toDateString())->get();

            foreach ($reservations as $reservation) {
                if ($reservation->customer_type == 'Guest' || $reservation->customer_type == 'guest') {
                    $reservation->customer = GuestCustomer::find($reservation->customer_id);
                } else if ($reservation->customer_type == 'registered' || $reservation->customer_type == 'Registered') {
                    $reservation->customer = Customer::find($reservation->customer_id);
                }
            }
            return response()->json($reservations, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function getDateRangeReservations(Request $request)
    {
        // dd($request->all());
        try {
            // $start_date = $request->input('start_date');
            // $end_date = $request->input('end_date');
            $search_type = $request->input('search_type');
            if ($search_type == 'reservation_no') {
                $query = Reservation::where('id', $request->reservation_number);
            } else {
                $query = Reservation::whereDate('created_at', '>=', $request->start_date)
                    ->whereDate('updated_at', '<=', $request->ending_date);

                if ($search_type == 'new_reservation') {
                    $query->where('status', '=', 'new_reservation');
                } elseif ($search_type == 'departures') {
                    $query->where('service_type_id', '=', 1)
                        ->where('airport_ride', '=', 'to_airport');
                } elseif ($search_type === 'departures_completed') {
                    $query->where('service_type_id', '=', 1)
                        ->where('airport_ride', '=', 'to_airport')
                        ->where('status', '=', 'Completed');
                } elseif ($search_type === 'arrivals') {
                    $query->where('service_type_id', '=', 1)
                        ->where('airport_ride', '=', 'from_airport');
                } elseif ($search_type === 'arrivals_completed') {
                    $query->where('service_type_id', '=', 1)
                        ->where('airport_ride', '=', 'from_airport')
                        ->where('status', '=', 'Completed');
                } elseif ($search_type === 'cancelled') {
                    $query->where('status', '=', 'Cancelled');
                } elseif ($search_type === 'completed') {
                    $query->where('status', '=', 'Completed');
                } elseif ($search_type === 'p_to_p') {
                    $query->where('service_type_id', '=', 3);
                } elseif ($search_type === 'p_to_p_completed') {
                    $query->where('service_type_id', '=', 3)
                        ->where('status', '=', 'Completed');
                } elseif ($search_type === 'cash_completed') {
                    $query->where('status', '=', 'Cash Completed');
                } elseif ($search_type === 'no_show') {
                    $query->where('status', '=', 'No Show');
                } elseif ($search_type === 'no_run') {
                    $query->where('status', '=', 'No Run');
                } elseif ($search_type === 'payment_type') {
                    $query->where('pay_type_id', '=', $request->payment_method); // Replace 'payment_type' with the actual column name
                } elseif ($search_type === 'company_id') {
                    $query->where('site_id', '=', $request->company_name); // Replace 'company_id' with the actual column name
                }
            }


            // elseif ($search_type === 'cc_4_digit') {
            //     $query->where('cc_last_4_digit', '=', $request->input('cc_4_digit')); // Replace 'cc_last_4_digit' with the actual column name
            // } elseif ($search_type === 'cc_number') {
            //     $query->where('cc_number', '=', $request->input('cc_number')); // Replace 'cc_number' with the actual column name
            // } elseif ($search_type === 'state_srch') {
            //     // Add logic to search by state/city/zipcode
            //     // You might need to customize this part based on your database schema
            //     $query->where(function ($subquery) use ($request) {
            //         $subquery->where('state', 'LIKE', '%' . $request->input('state_srch') . '%')
            //             ->orWhere('city', 'LIKE', '%' . $request->input('state_srch') . '%')
            //             ->orWhere('zipcode', 'LIKE', '%' . $request->input('state_srch') . '%');
            //     });
            // } elseif ($search_type === 'payment_account_id') {
            //     $query->where('payment_account_id', '=', $request->input('payment_account_id')); // Replace 'payment_account_id' with the actual column name
            // } elseif ($search_type === 'payment_account_number') {
            //     $query->where('payment_account_number', '=', $request->input('payment_account_number')); // Replace 'payment_account_number' with the actual column name
            // }

            $reservations = $query->get();
            // dd($reservations);

            foreach ($reservations as $reservation) {
                // Add logic to fetch customer data based on customer_type
                if ($reservation->customer_type == 'Guest' || $reservation->customer_type == 'guest') {
                    $reservation->customer = GuestCustomer::find($reservation->customer_id);
                } elseif ($reservation->customer_type == 'registered' || $reservation->customer_type == 'Registered') {
                    $reservation->customer = Customer::find($reservation->customer_id);
                }
            }

            return response()->json($reservations, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    // public function getDateRangeReservations(Request $request)
    // {
    //     try {
    //         $reservations = Reservation::whereDate('created_at', '>=', $request->start_date)
    //                 ->whereDate('updated_at', '<=', $request->ending_date)
    //                 ->get();

    //         foreach ($reservations as $reservation) {
    //             if ($reservation->customer_type == 'Guest' || $reservation->customer_type == 'guest') {
    //                 $reservation->customer = GuestCustomer::find($reservation->customer_id);
    //             }
    //             else if ($reservation->customer_type == 'registered' || $reservation->customer_type == 'Registered') {
    //                 $reservation->customer = Customer::find($reservation->customer_id);
    //             }
    //         }
    //         return response()->json($reservations, 200);
    //     } catch (\Exception $exception) {
    //         return response()->json(['error' => $exception->getMessage()], 500);
    //     }
    // }

    public function getAllReservationInfo()
    {
        try {
            $obj = new stdClass();

            $vehicles = Vehicle::select('id', 'vehicles_type')->get();
            $drivers = Driver::select('id', 'first_name', 'last_name')->get();

            $obj->vehicles = $vehicles;
            $obj->drivers = $drivers;

            return response()->json($obj, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function saveReservationWithUser(Request $request)
    {
        try {

            // dd($request->all());

            $existingCustomer = Customer::where('mobile_number', $request->customer_information['mobile_number'])
                ->orWhere('phone_number', $request->customer_information['phone_number'])
                ->orWhere('email', $request->customer_information['email'])
                ->first();

            if ($existingCustomer) {
                // Customer already exists, update the existing record
                $customer_obj = $existingCustomer;
            } else {
                // Customer does not exist, create a new record
                $customer_obj = new Customer();
            }

            $customer_obj->first_name = $request->customer_information['first_name'];
            $customer_obj->last_name = $request->customer_information['last_name'];
            $customer_obj->phone_number = $request->customer_information['phone_number'];
            $customer_obj->mobile_number = $request->customer_information['mobile_number'];
            $customer_obj->email = $request->customer_information['email'];
            $customer_obj->status = 'Registered Customer';

            if (isset($request->customer_information['password'])) {
                $customer_obj->password = bcrypt($request->customer_information['password']);
                $customer_obj->readable_password = $request->customer_information['password'];
            }

            $customer_obj->customer_type = 'Registered';

            if ($request->customer_information['pay_type'] == 3) {
                $customer_obj->card_type = $request->customer_information['card_type'];
                $customer_obj->cc_number = $request->customer_information['cc_number'];
                $customer_obj->csv_code = $request->customer_information['csv_code'];
                $customer_obj->billing_zip_code = $request->customer_information['billing_zip_code'];
                $customer_obj->card_holder_name = $request->customer_information['card_holder_name'];

                // Credit card expiration date format making
                $cc_expiry_month = $request->customer_information['cc_expiry_month'];
                $cc_expiry_year = $request->customer_information['cc_expiry_year'];

                $customer_obj->cc_expiry = $cc_expiry_month . '-' . $cc_expiry_year;
            } else if ($request->customer_information['pay_type'] == 4) {
                $customer_obj->direct_account_id = $request->customer_information['direct_account_id'];
            }

            $customer_obj->is_important_info = $request->customer_information['is_important_info'] ? $request->customer_information['is_important_info'] : 0;
            $customer_obj->is_discount_promotion = $request->customer_information['is_discount_promotion'] ? $request->customer_information['is_discount_promotion'] : 0;



            $customer_obj->save();

            if ($request->is_round_trip) {
                // For round trip reservations

                // Create first reservation
                $firstReservation = $this->store($request, $customer_obj->id, 'registered', $request->customer_information['pay_type'], false);
                // Create second reservation
                $secondReservation = $this->store($request, $customer_obj->id, 'registered', $request->customer_information['pay_type'], true);

                // Update round_trip column
                $firstReservation['round_trip'] = $secondReservation['id'];
                $secondReservation['round_trip'] = $firstReservation['id'];

                $firstReservation->save();
                $secondReservation->save();

                return response()->json([$firstReservation, $secondReservation, $customer_obj], 200);
            } else if ($request->is_schedule_round_trip) {
                // For schedule round trip reservations

                // Create first reservation
                $firstReservation = $this->store($request, $customer_obj->id, 'registered', $request->customer_information['pay_type'], false);

                // Create second reservation
                $secondReservation = $this->store($request, $customer_obj->id, 'registered', $request->customer_information['pay_type'], true);

                // Update round_trip column
                $firstReservation['round_trip'] = $secondReservation['id'];
                $secondReservation['round_trip'] = $firstReservation['id'];

                $firstReservation->save();
                $secondReservation->save();

                return response()->json([$firstReservation, $secondReservation, $customer_obj], 200);
            } else {
                //create single reservation

                $singleReservation = $this->store($request, $customer_obj->id, 'registered', $request->customer_information['pay_type'], false);

                return response()->json([$singleReservation, $customer_obj], 200);
            }
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function saveReservationWithGuestUser(Request $request)
    {
        try {
            // DB::beginTransaction();
            // dd($request->customer_information['card_type']);
            $existingGuestCustomer = GuestCustomer::where('mobile_number', $request->customer_information['mobile_number'])
                ->orWhere('phone_number', $request->customer_information['phone_number'])
                ->orWhere('email', $request->customer_information['email'])
                ->first();

            // dd($existingGuestCustomer);
            if ($existingGuestCustomer) {
                // Customer already exists, update the existing record
                $guest_customer = $existingGuestCustomer;
            } else {
                // Customer does not exist, create a new record
                $guest_customer = new GuestCustomer();
            }

            $guest_customer->first_name = $request->customer_information['first_name'];
            $guest_customer->last_name = $request->customer_information['last_name'];
            $guest_customer->phone_number = $request->customer_information['phone_number'];
            $guest_customer->mobile_number = $request->customer_information['mobile_number'];
            $guest_customer->email = $request->customer_information['email'];
            $guest_customer->status = 'Guest Customer';

            $guest_customer->customer_type = 'Guest';

            if ($request->customer_information['pay_type'] == 3) {

                $guest_customer->card_type = $request->customer_information['card_type'];
                $guest_customer->cc_number = $request->customer_information['cc_number'];
                $guest_customer->csv_code = $request->customer_information['csv_code'];
                $guest_customer->billing_zip_code = $request->customer_information['billing_zip_code'];
                $guest_customer->card_holder_name = $request->customer_information['card_holder_name'];

                // Credit card expiration date format making
                $cc_expiry_month = $request->customer_information['cc_expiry_month'];
                $cc_expiry_year = $request->customer_information['cc_expiry_year'];

                $guest_customer->cc_expiry = $cc_expiry_month . '-' . $cc_expiry_year;
            }

            // dd($guest_customer);
            $guest_customer->save();
            $payTypeId = $request->customer_information['pay_type'];

            // if ($request->is_round_trip) {
            //     // For round trip reservations

            //     //create first reservation
            //     $firstReservation = $this->store($request, $guest_customer->id, 'guest', false);

            //     //create second reservation
            //     $secondReservation = $this->store($request, $guest_customer->id, 'guest', true);

            //     // Update round_trip column
            //     $firstReservation['round_trip'] = $secondReservation['id'];
            //     $secondReservation['round_trip'] = $firstReservation['id'];

            //     $firstReservation->save();
            //     $secondReservation->save();

            //     return response()->json([$firstReservation, $secondReservation, $guest_customer], 200);
            // }
            // else if ($request->is_schedule_round_trip) {
            //     // For schedule round trip reservations

            //     // Create first reservation
            //     $firstReservation = $this->store($request, $guest_customer->id, 'guest', false);

            //     // Create second reservation
            //     $secondReservation = $this->store($request, $guest_customer->id, 'guest', true);

            //     // Update round_trip column
            //     $firstReservation['round_trip'] = $secondReservation['id'];
            //     $secondReservation['round_trip'] = $firstReservation['id'];

            //     $firstReservation->save();
            //     $secondReservation->save();

            //     return response()->json([$firstReservation, $secondReservation, $guest_customer], 200);
            // }
            // else {
            //     //create single reservation
            //     $singleReservation = $this->store($request, $guest_customer->id, 'guest', false);

            //     return response()->json([$singleReservation, $guest_customer], 200);
            // }
            if ($request->is_round_trip) {
                // For round trip reservations

                // Create first reservation
                $firstReservation = $this->store($request, $guest_customer->id, 'Guest', $payTypeId, false);

                // Create second reservation
                $secondReservation = $this->store($request, $guest_customer->id, 'Guest', $payTypeId, true);

                // Update round_trip column
                $firstReservation['round_trip'] = $secondReservation['id'];
                $secondReservation['round_trip'] = $firstReservation['id'];

                $firstReservation->save();
                $secondReservation->save();

                return response()->json([$firstReservation, $secondReservation, $guest_customer], 200);
            } else if ($request->is_schedule_round_trip) {
                // For schedule round trip reservations

                // Create first reservation
                $firstReservation = $this->store($request, $guest_customer->id, 'Guest', $payTypeId, false);

                // Create second reservation
                $secondReservation = $this->store($request, $guest_customer->id, 'Guest', $payTypeId, true);

                // Update round_trip column
                $firstReservation['round_trip'] = $secondReservation['id'];
                $secondReservation['round_trip'] = $firstReservation['id'];

                $firstReservation->save();
                $secondReservation->save();

                return response()->json([$firstReservation, $secondReservation, $guest_customer], 200);
            } else {
                //create single reservation
                // dd($payTypeId);
                $singleReservation = $this->store($request, $guest_customer->id, 'Guest', $payTypeId, false);

                return response()->json([$singleReservation, $guest_customer], 200);
            }
            // DB::commit();
        } catch (\Exception $exception) {
            // DB::rollback();
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function saveCustomerNotes(Request $request)
    {
        try {
            if ($request->customer_type == 'Registered' || $request->customer_type == 'registered') {
                $customer_obj = Customer::findOrFail($request->customer_id);
                $customer_obj->notes = $request->notes;
            }

            if ($request->customer_type == 'Guest' || $request->customer_type == 'guest') {
                $customer_obj = GuestCustomer::findOrFail($request->customer_id);
                $customer_obj->notes = $request->notes;
            }

            $customer_obj->save();

            $reservations = Reservation::where('customer_type', $request->customer_type)
                ->where('customer_id', $request->customer_id)
                ->get();
            foreach ($reservations as $reservation) {
                $reservation->customer_notes = $request->notes;
                $reservation->save();
            }
            // return response()->json($obj, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function updateDriverInfo(Request $request)
    {
        // dd($request->all());
        try {
            $blocked_driver = BlockDriver::where('driver_id', $request->driver_id)->where('customer_id', $request->customer_id)->where('is_active', 1)->first();

            if (!isset($blocked_driver)) {
                $reservation_obj = Reservation::findOrFail($request->reservation_id);

                $reservation_obj->driver_id = $request->driver_id;
                $reservation_obj->driver_status = null;

                $reservation_obj->save();

                emailDriver($request->driver_id, $request->reservation);

                return response()->json(['message' => 'Driver assigned successfully'], 201);
            } else {
                return response()->json('Driver Blocked for this Customer');
            }
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function getAllAirportData()
    {
        try {
            $obj = new stdClass();
            $obj->allAirports = Airport::select('id', 'name', 'airport_location')->where('is_active', 1)->get();

            return response()->json($obj, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function findReservation(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'confirm_number' => 'required|numeric',
            ]);

            if ($validator->fails()) {

                return response()->json(['error' => $validator->errors()], 422);
            } else {
                $reservation_obj = null;

                if ($request->confirm_number && $request->last_name) {

                    $customerId = Customer::where('last_name', '=', $request->last_name)
                        ->pluck('id')
                        ->first();

                    $guestCustomerId = GuestCustomer::where('last_name', '=', $request->last_name)
                        ->pluck('id')
                        ->first();

                    $customerType = Reservation::where('customer_id', $customerId)
                        ->pluck('customer_type')
                        ->first();

                    $guestCustomerType = Reservation::where('customer_id', $guestCustomerId)
                        ->pluck('customer_type')
                        ->first();

                    if ($customerType == 'Registered' || $customerType == 'registered') {
                        $reservation_obj = Reservation::where('id', '=', $request->confirm_number)
                            ->where('customer_id', '=', $customerId)
                            ->first();
                    } else if ($guestCustomerType == 'guest' || $guestCustomerType == 'Guest') {
                        $reservation_obj = Reservation::where('id', '=', $request->confirm_number)
                            ->where('customer_id', '=', $guestCustomerId)
                            ->first();
                    }
                } else if ($request->confirm_number && $request->mobile_number) {
                    $customerId = Customer::where('phone_number', '=', $request->mobile_number)
                        ->orWhere('mobile_number', $request->mobile_number)
                        ->pluck('id')
                        ->first();

                    $guestCustomerId = GuestCustomer::where('phone_number', '=', $request->mobile_number)
                        ->orWhere('mobile_number', $request->mobile_number)
                        ->pluck('id')
                        ->first();

                    $customerType = Reservation::where('customer_id', $customerId)
                        ->pluck('customer_type')
                        ->first();

                    $guestCustomerType = Reservation::where('customer_id', $guestCustomerId)
                        ->pluck('customer_type')
                        ->first();

                    if ($customerType == 'Registered' || $customerType == 'registered') {

                        $reservation_obj = Reservation::where('id', '=', $request->confirm_number)->where('customer_id', '=', $customerId)
                            ->latest('created_at')
                            ->first();
                    } else if ($guestCustomerType == 'guest' || $guestCustomerType == 'Guest') {

                        $reservation_obj = Reservation::where('id', '=', $request->confirm_number)->where('customer_id', '=', $guestCustomerId)
                            ->latest('created_at')
                            ->first();
                    }
                } else if ($request->confirm_number && $request->billing_zip_code) {
                    $customerId = Customer::where('billing_zip_code', '=', $request->billing_zip_code)
                        ->pluck('id')
                        ->first();

                    $guestCustomerId = GuestCustomer::where('billing_zip_code', '=', $request->billing_zip_code)
                        ->pluck('id')
                        ->first();

                    $customerType = Reservation::where('customer_id', $customerId)
                        ->pluck('customer_type')
                        ->first();

                    $guestCustomerType = Reservation::where('customer_id', $guestCustomerId)
                        ->pluck('customer_type')
                        ->first();

                    if ($customerType == 'Registered' || $customerType == 'registered') {

                        $reservation_obj = Reservation::where('id', '=', $request->confirm_number)->where('customer_id', '=', $customerId)
                            ->latest('created_at')
                            ->first();
                    } else if ($guestCustomerType == 'guest' || $guestCustomerType == 'Guest') {

                        $reservation_obj = Reservation::where('id', '=', $request->confirm_number)->where('customer_id', '=', $guestCustomerId)
                            ->latest('created_at')
                            ->first();
                    }
                }
                return response()->json($reservation_obj, 200);
            }
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }


    public function updateReservationStatus($reservationId, $reservationStatus = null)
    {
        try {
            $reservation_obj = Reservation::findOrFail($reservationId);

            $reservation_obj->status = $reservationStatus ? $reservationStatus : null;

            $reservation_obj->save();

            return response()->json(['message' => 'Reservation status saved successfully'], 201);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function updateCCStatus($customerId, $customerType)
    {
        try {
            if ($customerType == 'Registered' || $customerType == 'registered') {
                $customer_obj = Customer::findOrFail($customerId);

                if ($customer_obj->cc_status) {
                    $customer_obj->cc_status = null;
                } else {
                    $customer_obj->cc_status = 'decline';
                }
            }

            if ($customerType == 'Guest' || $customerType == 'guest') {
                $customer_obj = GuestCustomer::findOrFail($customerId);
                $customer_obj->cc_status = 'decline';
            }

            $customer_obj->save();

            return response()->json(['message' => 'Credit Card status update successfully'], 201);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function updateReservation(Request $request)
    {
        try {
            $pickUpDateTime = Carbon::parse($request->reservation['pick_up_date'] . ' ' . $request->reservation['pick_up_time']);
            $currentDateTime = Carbon::now();

            // Calculate the difference in hours between the pick-up date/time and current date/time
            $hoursDifference = $currentDateTime->diffInHours($pickUpDateTime);

            if ($hoursDifference > 12 || $request->is_admin) {
                // Less than 12 hours left for the reservation
                $reservation = Reservation::findOrFail($request->reservation['id']);

                $reservation->airport_ride = $request->reservation['airport_ride'];
                $reservation->departure_pick_up_date = $request->reservation['departure_pick_up_date'];
                $reservation->departure_pick_up_time = $request->reservation['departure_pick_up_time'];
                $reservation->distance = $request->reservation['distance'];
                $reservation->distance = $request->reservation['distance'];
                $reservation->drop_off_location = $request->reservation['drop_off_location'];
                $reservation->flight_number = $request->reservation['flight_number'];
                $reservation->home_drop_off_location = $request->reservation['home_drop_off_location'];
                $reservation->home_pick_up_location = $request->reservation['home_pick_up_location'];
                $reservation->luggage = $request->reservation['luggage'];
                $reservation->pick_up_date = $request->reservation['pick_up_date'];
                $reservation->pick_up_location = $request->reservation['pick_up_location'];
                $reservation->pick_up_time = $request->reservation['pick_up_time'];
                $reservation->price = $request->reservation['price'];
                $reservation->ride_type = $request->reservation['ride_type'];
                $reservation->round_trip = $request->reservation['round_trip'];
                // $reservation->schedule_service_type = $request->reservation['schedule_service_type'];
                $reservation->selected_airline = $request->reservation['selected_airline'];
                // $reservation->selected_airport = $request->reservation['selected_airport'];
                $reservation->selected_flight_destination = $request->reservation['selected_flight_destination'];
                $reservation->service_type_id = $request->reservation['service_type_id'];
                $reservation->time_duration_text = $request->reservation['time_duration_text'];
                // $reservation->timely_millage_charges = $request->reservation['timely_millage_charges'];
                $reservation->additional_stops = $request->reservation['additional_stops'];
                // $reservation->traveller_types_charges = $request->reservation['traveller_types_charges'];
                $reservation->travellers = $request->reservation['travellers'];
                $reservation->travellers_types = $request->reservation['travellers_types'];
                $reservation->vehicle_type_id = $request->reservation['vehicle_type_id'];
                $reservation->vehicles_count = $request->reservation['vehicles_count'];
                $reservation->drop_off_hours = $request->reservation['drop_off_hours'];

                if ($request->reservation['service_type_id'] !== 4 && isset($request->reservation['selected_service'])) {

                    if ($request->reservation['selected_service']['ride_type'] == 'millage') {
                        // If service selected for millage ride type
                        $reservation->ride_type = $request->reservation['selected_service']['ride_type'];
                        $reservation->additional_millage_charges = $request->reservation['selected_service']['millage_service']['additional_millage_charges'];
                        $reservation->timely_millage_charges = $request->reservation['selected_service']['millage_service']['timely_millage_charges'];
                        $reservation->traveller_types_charges = $request->reservation['selected_service']['millage_service']['traveller_types_charges'];
                        $reservation->price = $request->reservation['selected_service']['millage_service']['est_with_add_fare'];
                        // $reservation->price = $request->reservation['selected_service']['millage_service']['est_with_add_fare'];
                    } else if ($request->reservation['selected_service']['ride_type'] == 'hourly') {
                        // If service selected for hourly ride type
                        $reservation->ride_type = $request->reservation['selected_service']['ride_type'];
                        $reservation->additional_millage_charges = $request->reservation['selected_service']['hourly_service']['additional_hourly_charges'];
                        $reservation->timely_millage_charges = $request->reservation['selected_service']['hourly_service']['timely_hourly_charges'];
                        $reservation->traveller_types_charges = $request->reservation['selected_service']['hourly_service']['traveller_types_charges'];
                        $reservation->price = $request->reservation['selected_service']['hourly_service']['est_with_add_fare'];
                    }
                    $reservation->save();

                    return response()->json(['message' => 'Reservation updated successfully'], 201);
                }
            } else {
                // More than or equal to 12 hours left for the reservation
                return response()->json(['message' => 'Reservation Cannot be updated now'], 403);
            }
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function updateScheduleReservation(Request $request)
    {
        try {
            // dd($request->all(), $request->form_data);

            // if(isset($request->reservation['pick_up_date'])){
            //     $pickUpDateTime = Carbon::parse($request->reservation['pick_up_date'] . ' ' . $request->reservation['pick_up_time']);
            //     $currentDateTime = Carbon::now();
            // }
            // else
            if (isset($request->form_data['pick_up_date'])) {
                $pickUpDateTime = Carbon::parse($request->form_data['pick_up_date'] . ' ' . $request->reservation['pick_up_time']);
                $currentDateTime = Carbon::now();
            }
            if (isset($request->form_data['departure_pick_up_date'])) {
                $pickUpDateTime = Carbon::parse($request->form_data['departure_pick_up_date'] . ' ' . $request->reservation['departure_pick_up_time']);
                $currentDateTime = Carbon::now();
            }

            // Calculate the difference in hours between the pick-up date/time and current date/time
            $hoursDifference = $currentDateTime->diffInHours($pickUpDateTime);

            if ($hoursDifference > 12 || $request->is_admin) {
                // Less than 12 hours left for the reservation
                $reservation_obj = Reservation::findOrFail($request->reservation['id']);

                // dd($request->new_selected_service['shuttle_name']);

                $reservation_obj->shuttle_name = isset($request->new_selected_service['shuttle_name']) ? $request->new_selected_service['shuttle_name'] : $reservation_obj->shuttle_name;
                // $reservation_obj->schedule_service_type = $request->reservation['schedule_service_type'];
                // $reservation_obj->vehicle_type_id = $request->reservation['vehicle_type_id'];
                $reservation_obj->departure_pick_up_date = $request->form_data['departure_pick_up_date'] ?? null;
                $reservation_obj->departure_pick_up_time = $request->form_data['departure_pick_up_time'] ?? null;
                $reservation_obj->pick_up_date = $request->form_data['pick_up_date'];

                // dd($reservation_obj);

                $reservation_obj->pick_up_time = $request->form_data['home_pick_up_time'];
                $reservation_obj->drop_off_time = $request->reservation['drop_off_time'];

                $reservation_obj->travellers = $request->reservation['travellers'];
                $reservation_obj->luggage = $request->reservation['luggage'];
                $reservation_obj->check_in_bags = $request->reservation['check_in_bags'];
                $reservation_obj->pick_up_location = isset($request->new_selected_service['pick_up_location']) ? $request->new_selected_service['pick_up_location'] : $reservation_obj->pick_up_location;
                $reservation_obj->drop_off_location = isset($request->new_selected_service['drop_off_location']) ? $request->new_selected_service['drop_off_location'] : $reservation_obj->drop_off_location;
                $reservation_obj->selected_airline = $request->reservation['selected_airline'] ? $request->reservation['selected_airline'] : null;
                $reservation_obj->selected_flight_destination = $request->reservation['selected_flight_destination'] ? $request->reservation['selected_flight_destination'] : null;
                $reservation_obj->flight_number = $request->reservation['flight_number'] ? $request->reservation['flight_number'] : null;

                // dd($reservation_obj);

                if (isset($request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_location']) && isset($request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_location']['address'])) {
                    $reservation_obj->home_pick_up_location = $request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_location']['address'];
                }
                if (isset($request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_location']) && isset($request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_location']['address'])) {
                    $reservation_obj->home_drop_off_location = $request->reservation['scheduleServiceAdditionInformation']['schedule_drop_off_location']['address'];
                }

                if ($request->reservation['ride_type'] == 'airport' && $request->reservation['airport_ride_type'] == 'to_airport') {
                    $reservation_obj->airport_ride = $request->reservation['airport_ride_type'];
                    if (isset($request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address']) && isset($request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address']['address'])) {
                        $reservation_obj->home_pick_up_location = $request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address'];
                    }
                }
                if ($request->reservation['ride_type'] == 'airport' && $request->reservation['airport_ride_type'] == 'from_airport') {
                    $reservation_obj->airport_ride = $request->reservation['airport_ride_type'];
                    if (isset($request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address']) && isset($request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address']['address'])) {
                        $reservation_obj->home_drop_off_location = $request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address'] !== '' ? $request->reservation['scheduleServiceAdditionInformation']['schedule_pick_up_or_drop_off_location']['address'] : '';
                    }
                }

                $reservation_obj->vehicle_capacity = $request->reservation['vehicle_capacity'] ? $request->reservation['vehicle_capacity'] : null;

                $reservation_obj->distance = $request->reservation['distance'] ? $request->reservation['distance'] : null;
                $reservation_obj->time_duration_text = $request->reservation['time_duration_text'] ? $request->reservation['time_duration_text'] : null;
                // $reservation_obj->time_duration_hours = $request->reservation['time_duration_hours'] ? $request->reservation['time_duration_hours'] : null;

                $reservation_obj->additional_millage_charges = $request->reservation['additional_millage_charges'];
                $reservation_obj->timely_millage_charges = $request->reservation['timely_millage_charges'];
                $reservation_obj->price = isset($request->new_selected_service['price_per_person']) ? $request->new_selected_service['price_per_person'] : $reservation_obj->price;
                $reservation_obj->fare_per_person = isset($request->new_selected_service['price_per_person']) ? $request->new_selected_service['price_per_person'] : $reservation_obj->fare_per_person;
                $reservation_obj->fare_with_add_charges = $request->reservation['fare_with_add_charges'];

                // dd($reservation_obj);

                $reservation_obj->save();

                return response()->json(['message' => 'Reservation updated successfully'], 201);
            } else {
                // More than or equal to 12 hours left for the reservation
                return response()->json(['message' => 'Reservation Cannot be updated now'], 403);
            }
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function updateCustomer(Request $request)
    {

        try {
            $customer = Customer::find($request->customer_data['id']);

            $customer->first_name = $request->customer_data['first_name'];
            $customer->last_name = $request->customer_data['last_name'];
            $customer->phone_number = $request->customer_data['phone_number'];
            $customer->mobile_number = $request->customer_data['mobile_number'];
            $customer->email = $request->customer_data['email'];
            $customer->card_type = $request->customer_data['card_type'];
            $customer->cc_number = $request->customer_data['cc_number'];
            $customer->cc_expiry = $request->customer_data['cc_expiry'];
            $customer->csv_code = $request->customer_data['csv_code'];
            $customer->billing_zip_code = $request->customer_data['billing_zip_code'];
            $customer->card_holder_name = $request->customer_data['card_holder_name'];
            // $customer->billing_address = $request->customer_data['billing_address']['address'] ? $request->customer_data['billing_address']['address'] : null;
            $customer->billing_phone_number = $request->customer_data['billing_phone_number'];

            $customer->save();

            return response()->json(['message' => 'Customer update successfully'], 201);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function blockDriver(Request $request)
    {

        $blocked_driver = BlockDriver::where('customer_id', $request->customer_id)->where('driver_id', $request->driver_id)->first();

        if (!isset($blocked_driver)) {
            $block_driver = new BlockDriver();

            $block_driver->user_id = Auth::id();
            $block_driver->customer_id = $request->customer_id;
            $block_driver->driver_id = $request->driver_id;
            $block_driver->reservation_id = $request->reservation_id;
            $block_driver->reason = $request->reason;

            $block_driver->save();

            return response()->json(['message' => 'Driver Blocked successfully'], 201);
        } else if (isset($blocked_driver) && $blocked_driver->is_active === 0) {
            $blocked_driver->is_active = 1;
            $blocked_driver->save();
            return response()->json(['message' => 'Driver Blocked successfully'], 201);
        } else {
            return response()->json(['message' => 'Driver already Blocked for this Customer'], 201);
        }
    }

    public function blockedDriversIndex()
    {
        $blocked_drivers = BlockDriver::all();

        return view('admin.blocked-drivers.index', [
            'blockedDrivers' => $blocked_drivers
        ]);
    }

    public function getFormData()
    {
        $obj = new stdClass();

        $obj->allDrivers = Driver::select('id', 'category_id', 'first_name', 'last_name')->get();
        $obj->allCustomers = Customer::select('id', 'first_name', 'last_name')->get();

        return response()->json($obj, 200);
    }

    public function updateIsActive(Request $request, $id)
    {
        try {
            dd($request->status);
            BlockDriver::whereId($id)->update([
                'is_active' => $request->status
            ]);

            return response()->json(['message' => 'Blocked Driver Status updated successfully'], 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function checkCouponValidation(Request $request)
    {
        $vehicleTypeId = $request->input('vehicle_type_id');

        $coupon = Coupon::select('id', 'coupon_type', 'coupon_price')
            ->where('coupon_code', $request->input('coupon_code'))
            ->where('city_id', $request->input('city_id'))
            ->where('service_id', $request->input('service_type_id')) //Add where clause fro checking service type e.g Airport, Hourly etc
            ->whereJsonContains('vehicle_make_ids', $vehicleTypeId) // Add whereJsonContains clause for vehicle type
            ->where('expiry_date', '>=', $request->input('pick_up_date')) // Add whereDate clause to check expiry date with pickup date
            ->where('is_active', 1)
            ->first();

        return response()->json($coupon, 200);
    }

    public function getData()
    {
        try {
            $obj = new stdClass();

            $obj->pay_types = PayType::select('id', 'name')
                ->where('is_active', 1)
                ->get();

            $obj->direct_accounts = DirectAccount::select('id', 'account_number', 'account_name')
                ->where('is_active', 1)
                ->get();

            return response()->json($obj, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function editFlightInformation($id, Request $request)
    {

        // dd($request->all());
        try {
            $reservation = Reservation::findOrFail($id);
            $reservation->selected_airline = $request->airline;
            $reservation->flight_number = $request->flight_number;
            $reservation->selected_flight_destination = $request->flight_destination;

            // dd($reservation);
            $reservation->save();
            return response()->json(['message' => 'Edit Flight Information']);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
    public function editFlightDateTimeInformation($id, Request $request)
    {

        // dd($request->all());
        try {
            $reservation = Reservation::findOrFail($id);
            $reservation->pick_up_time = $request->pick_up_time;
            $reservation->pick_up_date = $request->pick_up_date;
            $reservation->departure_pick_up_date = $request->departure_pick_up_date;
            $reservation->departure_pick_up_time = $request->departure_pick_up_time;

            // dd($reservation);
            $reservation->save();
            return response()->json(['message' => 'Edit Flight Information']);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function getEditFlightInformation()
    {
        try {
            $obj = new stdClass();
            $obj->allAirlines = Airline::select('id', 'airline_name')->where('is_active', 1)->get();
            $obj->allFlightDestinations = FlightDestination::select('id', 'name', 'code')->where('is_active', 1)->get();
            return response()->json($obj, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
}
