<?php

namespace App\Http\Controllers\Admin\Reports\ProfitLoss;

use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Models\Admin\Reservation;
use App\Http\Controllers\Controller;
use App\Models\Admin\ExpenseDetails;

class ProfitLossReportController extends Controller
{
    public function index()
    {
        return view('admin.reports.profit-loss.index');
    }

    public function getFilterData(Request $request)
    {
        try {
            $fromDate = $request->input('fromDate');
            $toDate = $request->input('toDate');

            // Retrieve reservation data within the specified date range
            $income_data = Reservation::where('driver_status', 'Completed')
            ->select('total_fare', 'pick_up_date')
            ->whereDate('pick_up_date', '>=', $fromDate)
                ->whereDate('pick_up_date', '<=', $toDate)
                ->get();

            // Retrieve expense data within the specified date range
            $expense_data = ExpenseDetails::select('amount', 'expense_date')
            ->whereDate('expense_date', '>=', $fromDate)
                ->whereDate('expense_date', '<=', $toDate)
                ->get();

            // Generate a list of dates within the specified range
            $dateRange = [];
            $currentDate = Carbon::parse($fromDate);
            $endDate = Carbon::parse($toDate);

            while ($currentDate->lte($endDate)) {
                $dateRange[] = $currentDate->format('Y-m-d');
                $currentDate->addDay();
            }

            // Group income data by pick_up_date and calculate the sum of prices for each date
            $groupedIncome = $income_data->groupBy('pick_up_date')->map(function ($group) {
                return $group->sum('total_fare');
            });

            // Group expense data by expense_date and calculate the sum of amounts for each date
            $groupedExpenses = $expense_data->groupBy('expense_date')->map(function ($group) {
                return $group->sum('amount');
            });

            // Merge income, expense, and profit/loss data based on pick_up_date
            $mergedData = collect();

            foreach ($dateRange as $date) {
                $income = $groupedIncome[$date] ?? 0;
                $expenses = $groupedExpenses[$date] ?? 0;
                $profitLoss = $income - $expenses;

                $mergedData->push([
                    'date' => $date,
                    'income' => $income,
                    'expense' => $expenses,
                    'profit_loss' => $profitLoss,
                ]);
            }

            return response()->json([
                'filtered_data' => $mergedData,
            ], 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }
}
