<?php

namespace App\Http\Controllers\Admin;


use Carbon\Carbon;
use App\Models\Module;
use Carbon\CarbonPeriod;
use App\Models\Admin\City;
use App\Models\Admin\Site;
use App\Models\Admin\Driver;
use Illuminate\Http\Request;
use App\Models\Admin\Airport;
use App\Models\Admin\Country;
use App\Models\Admin\Customer;
use App\Models\Admin\Employee;
use App\Models\Admin\Services;
use App\Models\Admin\Reservation;
use Illuminate\Support\Facades\DB;
use App\Models\Admin\GuestCustomer;
use App\Http\Controllers\Controller;
use App\Models\Admin\ExpenseDetails;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;

class AdminHomeController extends Controller
{
    public function afterLogin()
    {
        return view('admin.after_login');
    }
    public function dashboard()
    {
        // dd('here');
        $user=Auth::user();
        if($user->role_id == 1)
        {
            return view('admin.dashboard');
        }
        else{
            return view('admin.after_login');
        }
    }

    public function userProfilePage()
    {
        $user_profile=Employee::select('first_name', 'last_name', 'email_address', 'dob', 'address', 'cell_phone')->where('user_id', Auth::id())->first();
        return view('admin.user-profile', [
            'user' => session('user'),
            'user_profile' => $user_profile
        ]);
    }

    public function getReservationCountForLineChart($dateRange)
    {
        try {
            $dates = [];
            $reservationCounts = [];
            $monthNames = [];
            $startDate = null;
            $endDate = Carbon::now();

            // Determine the start date and step based on the date range
            switch ($dateRange) {
                case 'week-ago':
                    $startDate = Carbon::now()->subDays(7);
                    $step = '1 day';
                    break;
                case 'last-30-days':
                    $startDate = Carbon::now()->subMonth();
                    $step = '1 day';
                    break;
                case 'last-year':
                    $startDate = Carbon::now()->startOfYear();
                    $step = '1 month';
                    break;
                default:
                    return response()->json(['error' => 'Invalid date range'], 400);
            }

            // Generate the date period
            $period = CarbonPeriod::create($startDate, $step, $endDate);

            foreach ($period as $date) {
                if ($dateRange === 'last-year') {
                    $monthName = $date->format('F');
                    $reservationCounts[] = Reservation::whereYear('created_at', $date->year)
                        ->whereMonth('created_at', $date->month)
                        ->where('status', '!=','Cancelled')
                        ->count();
                    $dates[] = $date->format('Y-m-d');
                    $monthNames[] = $monthName;
                } else {
                    $formattedDate = $date->format('Y-m-d');
                    $reservationCounts[] = Reservation::whereDate('created_at', $formattedDate)
                        ->where('status', '!=','Cancelled')
                        ->count();
                    $dates[] = $date->format('d-m');
                }
            }

            $response = [
                'dates' => $dates,
                'reservation_counts' => $reservationCounts,
            ];

            // Include month names for the 'last-year' range
            if ($dateRange === 'last-year') {
                $response['month_names'] = $monthNames;
            }

            return $response;
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function getIncomeExpenseForLineChart()
    {
        try {
        $startDate = now()->startOfYear();
        $endDate = now();
        $dates = [];
        $reservationAmount = [];
        $monthNames = [];
        $period = CarbonPeriod::create($startDate, '1 month', $endDate);

        foreach ($period as $date) {
            $formattedDate = $date->format('Y-m-d');
            $monthName = $date->format('F');

            $reservationAmount[] =
                Reservation::where('driver_status', 'Completed')
                ->whereYear('pick_up_date', '=', $date->year)
                ->whereMonth('pick_up_date', '=', $date->month)
                ->sum('total_fare');

            $dates[] = $formattedDate;
            $monthNames[] = $monthName;
        }

        return [
            'dates' => $dates,
            'reservation_amounts' => $reservationAmount,
            'month_names' => $monthNames
        ];
        }
        catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }

    }
    public function getProfitLoss($dateRange)
    {
        $now = Carbon::now();
        $incomeQuery = Reservation::where('driver_status', 'Completed');
        $expenseQuery = ExpenseDetails::query();

        switch ($dateRange) {
            case 'last_seven_days':
                $startDate = $now->subDays(7);
                break;
            case 'this_month':
                $startDate = $now->startOfMonth();
                break;
            case 'this_year':
                $incomeQuery->whereYear('pick_up_date', $now->year);
                $expenseQuery->whereYear('expense_date', $now->year);
                return $this->calculateProfitLoss($incomeQuery, $expenseQuery);
            default:
                return [
                    'income' => 0,
                    'expense' => 0,
                    'balance' => 0
                ];
        }

        $incomeQuery->whereDate('pick_up_date', '>=', $startDate)
            ->whereDate('pick_up_date', '<', $now);
        $expenseQuery->whereDate('expense_date', '>=', $startDate)
            ->whereDate('expense_date', '<', $now);

        return $this->calculateProfitLoss($incomeQuery, $expenseQuery);
    }

    private function calculateProfitLoss($incomeQuery, $expenseQuery)
    {
        $incomeData = $incomeQuery->sum('total_fare');
        $expenseData = $expenseQuery->sum('amount');
        $profitLoss = $incomeData - $expenseData;

        return [
            'income' => $incomeData,
            'expense' => $expenseData,
            'balance' => $profitLoss
        ];
    }

    // public function getProfitLoss($dateRange)
    // {
    //     try {


    //         if ($dateRange == 'last_seven_days') {

    //             $income_data = Reservation::where('driver_status', 'Completed')
    //             ->whereDate('pick_up_date', '>=', Carbon::now()->subDays(7))
    //                 ->whereDate('pick_up_date', '<', Carbon::now())
    //                 ->sum('total_fare');

    //             $expense_data = ExpenseDetails::whereDate('expense_date', '>=', Carbon::now()->subDays(7))
    //                 ->whereDate('expense_date', '<', Carbon::now())
    //                 ->sum('amount');
    //             $profitLoss = $income_data - $expense_data;
    //             return [
    //                 'income' => $income_data,
    //                 'expense' => $expense_data,
    //                 'balance' => $profitLoss
    //             ];
    //         }
    //         else if ($dateRange == 'this_month') {

    //             $income_data = Reservation::where('driver_status', 'Completed')
    //                 ->whereDate('pick_up_date', '>=', Carbon::now()->startOfMonth())
    //                 ->whereDate('pick_up_date', '<', Carbon::now())
    //                 ->sum('price');

    //             $expense_data = ExpenseDetails::whereDate('expense_date', '>=', Carbon::now()->startOfMonth())
    //                 ->whereDate('expense_date', '<', Carbon::now())
    //                 ->sum('amount');
    //             $profitLoss = $income_data - $expense_data;
    //             return [
    //                 'income' => $income_data,
    //                 'expense' => $expense_data,
    //                 'balance' => $profitLoss
    //             ];
    //         }
    //         else if ($dateRange == 'this_year') {
    //             $income_data = Reservation::where('driver_status', 'Completed')
    //                 ->whereYear('pick_up_date', '=', now()->year)
    //                 ->sum('price');

    //             $expense_data = ExpenseDetails::whereYear('expense_date', '=', now()->year)
    //                 ->sum('amount');

    //             $profitLoss = $income_data - $expense_data;
    //             return [
    //                 'income' => $income_data,
    //                 'expense' => $expense_data,
    //                 'balance' => $profitLoss
    //             ];
    //         }
    //     } catch (\Exception $exception) {
    //         return response()->json(['error' => $exception->getMessage()], 500);
    //     }

    // }
    public function getData()
    {
        try {
            $reservations = Reservation::select('id', 'pick_up_date', 'service_type_id')
        ->where('driver_status', '!=', 'Completed')
        ->orWhereNull('driver_status')
        ->get();


        $aggregatedData = [];

        foreach ($reservations as $reservation) {
            $pickUpDate = $reservation->pick_up_date;
            $serviceTypeId = $reservation->service_type_id;

            $key = "$pickUpDate-$serviceTypeId";

            if (!isset($aggregatedData[$key])) {
                $service = Services::where('id', $serviceTypeId)->first();

                $serviceName = $service ? $service->service_name : 'Unknown';

                $cleanServiceName = str_replace("\n", '', $serviceName);

                $serviceNameParts = explode('/', $cleanServiceName);
                $shortServiceName = end($serviceNameParts);

                if (count($serviceNameParts) === 1) {
                    $shortServiceName = $cleanServiceName;
                }

                $aggregatedData[$key] = [
                    'pick_up_date' => $pickUpDate,
                    'service_type_id' => $serviceTypeId,
                    'service_name' => $shortServiceName,
                    'total_reservations' => 1,
                ];
            }
            else {
                // If the key exists, increment the total_reservations count
                $aggregatedData[$key]['total_reservations']++;
            }
        }
        // dd($aggregatedData);
        $result = array_values($aggregatedData);

        return response()->json($result, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }

    }
    public function cardData()
    {
        try {

            $customers = Customer::where(function($q){
                            $q->where('customer_type', 'Registered')
                            ->orWhere('customer_type', 'registered');
                        })->count();
            $guestCustomers =   Customer::where(function($q){
                                $q->where('customer_type', 'Guest')
                                ->orWhere('customer_type', 'guest');
                            })->count();
            $services = Services::where('is_active', 1)->count();
            $sites = Site::where('is_active', 1)->count();
            $inactiveServices = Services::where('is_active', 0)->count();
            $inactiveSites = Site::where('is_active', 0)->count();
            $inactiveCountries = Country::where('is_active', 0)->count();
            $inActiveDrivers = Driver::where('is_active', 0)->count();
            $inActiveCities = City::where('is_active', 0)->count();
            $inActiveAirports = Airport::where('is_active', 0)->count();
            $incomeData = Reservation::where('driver_status', 'Completed')->sum('total_fare');
            $expenseData = ExpenseDetails::sum('amount');

            $cardData = [
                'customers_count' => $customers,
                'guest_customer_count' => $guestCustomers,
                'services_count' => $services,
                'inactive_services_count' => $inactiveServices,
                'in_active_drivers' => $inActiveDrivers,
                'sites_count' => $sites,
                'all_incomes' => $incomeData,
                'all_expenses' => $expenseData,
                'in_active_countries' => $inactiveCountries,
                'in_active_cities' => $inActiveCities,
                'in_active_airports' => $inActiveAirports,
                'in_active_sites' => $inactiveSites,
            ];
            return response()->json($cardData, 200);
        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }

    }
    public function userProfileUpdate(Request $request)
    {
        try {
            DB::beginTransaction();
            $user = Auth::user();

            Employee::where('user_id', $user->id)->update([
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'dob' => $request->dob,
                'cell_phone' => $request->cell_phone,
                'address' => $request->address
            ]);

            DB::commit();
            return response()->json(['message' => 'Profile updated successfully'], 200);
        } catch (\Exception $exception) {
            DB::rollBack();
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }


    public function changePasswordPage()
    {
        return view('admin.change-password', [
            'user' => session('user'),
        ]);
    }
    public function checkPassword(Request $request)
    {
        if (Hash::check($request->old_password, Auth::user()->getAuthPassword())) {
            return response()->json(true);
        } else {
            return response()->json(false);
        }
    }
    public function updatePassword(Request $request)
    {
        try {
            DB::beginTransaction();

            $user = Auth::user();

            if (Hash::needsRehash($request->password)) {
                $user->password = Hash::make($request->password);
                Auth::logout();
            }

            $user->save();

            DB::commit();
            return response()->json(['message' => 'Password updated successfully']);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
